require('dotenv').config();
const assert = require('assert');
const { createServer } = require('http');
const { WebSocketServer } = require('ws');
const { createEndpoint } = require('@jambonz/node-client-ws');
const express = require('express');
const { ejecutarLlamadas } = require('./utils/call-launcher');
const logger = require('./utils/logger')('app-main');
const yardmaster = require('./utils/yardmaster');
const { iniciarScheduler } = require('./scheduler/bot-scheduler');
const { startHeartbeat } = require('./utils/monitor-hub');


// Configuración de puertos
const HTTP_PORT = process.env.HTTP_PORT || 3000;
const WS_PORT = process.env.WS_PORT || 8765;

// Validar variables de entorno
assert.ok(process.env.ULTRAVOX_API_KEY, 'ULTRAVOX_API_KEY es requerido');
assert.ok(process.env.JAMBONZ_ACCOUNT_SID, 'JAMBONZ_ACCOUNT_SID es requerido');

const app = express();
app.use(express.json());
startHeartbeat();   // después de crear wssStream/wsServer
// ====================
// 1. CONFIGURACIÓN HTTP
// ====================

// Middleware de logs
app.use((req, res, next) => {
  logger.info(`HTTP ${req.method} ${req.path}`, { ip: req.ip });
  next();
});

app.use(express.json({ type: ['application/json','application/*+json'], limit: '10mb' }));
app.use(express.urlencoded({ extended: true }));


// Ruta raíz
app.get('/', (req, res) => {
  res.status(200).json({
    status: 'running',
    services: {
      http: `http://0.0.0.0:${HTTP_PORT}`,
      websocket: `ws://0.0.0.0:${WS_PORT}`,
      environment: process.env.NODE_ENV || 'development'
    },
    timestamp: new Date().toISOString()
  });
});




// IMPORTANTE: Agregar router de transferencia directamente
//const transferRouter = require('./routes/http/transfer');
//app.use('/transfer', transferRouter); // ✅ CORRECCIÓN: Agregar esta lí

// Rutas HTTP
require('./routes/http')(app);


// Endpoint de estado
app.post('/status', (req, res) => {
  const { call_sid, call_status } = req.body;
  logger.info({ call_sid, call_status }, 'Estado de llamada recibido');
  res.sendStatus(200);
});

// ======================
// 2. CONFIGURACIÓN WEBSOCKET
// ======================

const wssStream = new WebSocketServer({ noServer: true });
const wsServer = createServer();

// Configuración específica de Jambonz
const makeService = createEndpoint({
  server: wsServer,
  externalWss: [
  { path: '/audio-stream', wss: wssStream }, 
  { path: '/monitor',      wss: wssStream } 
 ]
});



// Configuración de rutas WebSocket
require('./routes/ws')({ 
  logger, 
  makeService, 
  wssStream 
});

// Manejador de conexiones WS para audio-stream
wssStream.on('connection', (ws, req) => {

  const url = new URL(req.url, `http://${req.headers.host}`);
  const pathname = url.pathname;
  //  Solo exigir callSid cuando es el WS de audio:
  if (pathname !== '/audio-stream') return;
  const query = url.searchParams;
  const callSid = query.get('callSid');
  const parentCallSid = query.get('parentCallSid');

  if (!callSid) {
    logger.warn('Conexión WS rechazada: falta callSid', { url: req.url });
    return ws.close(1003, 'Missing callSid');
  }

  logger.info('Nueva conexión WebSocket', { callSid, parentCallSid });
  yardmaster.addJambonzWebsocket(ws, callSid, parentCallSid);

  ws.on('close', () => {
    logger.info('Conexión WebSocket cerrada', { callSid });
  });
});

// ======================
// 3. INICIO DE SERVIDORES
// ======================

const startServers = async () => {
  try {
    // Servidor HTTP
    app.listen(HTTP_PORT, '0.0.0.0', () => {
      logger.info(`Servidor HTTP iniciado en http://0.0.0.0:${HTTP_PORT}`);
    });

    // Servidor WebSocket
    wsServer.listen(WS_PORT, '0.0.0.0', () => {
      logger.info(`Servidor WebSocket iniciado en ws://0.0.0.0:${WS_PORT}`);
    });

    // Iniciar scheduler
    iniciarScheduler();
    logger.info('Scheduler de llamadas iniciado');

    // Opcional: Ejecutar llamadas iniciales
    await ejecutarLlamadas();
  } catch (err) {
    logger.error('Error al iniciar servidores:', { error: err.message });
    process.exit(1);
  }
};

// Manejo de cierre limpio
process.on('SIGINT', () => {
  logger.info('Cerrando servidores...');
  wsServer.close();
  process.exit(0);
});

// Iniciar la aplicación
startServers();