// routes/http/upload.js
const express = require('express');
const multer = require('multer');
const fs = require('fs');
const fsp = fs.promises;
const path = require('path');
const xlsx = require('xlsx');
const { v4: uuidv4 } = require('uuid');

const storage = require('../../utils/storage'); // Redis helper (ajusta si tu ruta es distinta)

const router = express.Router();

// ---------------- Multer (idntico a tu configuracin) ----------------
const storageDisk = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = 'uploads/';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueName = `${Date.now()}-${Math.round(Math.random() * 1E9)}-${file.originalname}`;
    cb(null, uniqueName);
  }
});

const upload = multer({
  storage: storageDisk,
  limits: {
    fileSize: 15 * 1024 * 1024,
    files: 1
  },
  fileFilter: (req, file, cb) => {
    const allowed = new Set([
      'text/csv',
      'application/csv',
      'application/vnd.ms-excel',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    ]);
    if (allowed.has(file.mimetype)) return cb(null, true);
    return cb(new Error('Tipo de archivo no soportado'), false);
  }
});

// ---------------- Paths de salida ----------------
const DEST_DIR = path.join(__dirname, '../../data');
const DEST_CSV = path.join(DEST_DIR, 'contactos.csv');
const DEST_JSON = path.join(DEST_DIR, 'contactos_ext.json');

// ---------------- Helpers ----------------
const stripAccents = (s = '') => String(s).normalize('NFD').replace(/[\u0300-\u036f]/g, '');
const normKey = (s = '') => stripAccents(String(s)).toLowerCase().replace(/[^a-z0-9]+/g, '').trim();

const SYN = {
  telefono: ['telefono','telfono','numero','nmero','celular','movil','mvil','phone','msisdn','numcel','telf','phone_number','numero_telefono'].map(normKey),
  identificacion: ['identificacion','identificacin','dni','documento','doc','id','cedula','ci','ruc','rut','ssn','nrodoc','numdoc','docid','numero_documento'].map(normKey),
  nombrecompleto: ['nombrecompleto','nombre','nombres','apellidos','cliente','contacto','fullname','full_name','nombresyapellidos'].map(normKey),
  mejor_perfil: ['mejorperfil','perfil','segmento','scoreperfil'].map(normKey),
  origen: ['origen','fechaorigen','fuentedato','source'].map(normKey),
  dias_mora: ['diasmora','mora','dias_en_mora','diasatraso'].map(normKey),
  monto_inicial: ['montoinicial','monto','deuda','importe','saldo','amount','balance'].map(normKey),
  mod_init_cta: ['modinitcta','modinicta','modalidad','producto','plancomercial'].map(normKey),
  referencia: ['referencia','ref','expediente','caso','reference'].map(normKey),
  plan: ['plan','tarifa','paquete'].map(normKey),
  fecha_vencimiento: ['fechavencimiento','vencimiento','fechalimite','duedate','expiration'].map(normKey),
  tipo_obligacion: ['tipoobligacion','tipo','clase'].map(normKey),
  descuento: ['descuento','dscto','rebaja','discount'].map(normKey),
  min: ['min','MIN','minimum','minimo'].map(normKey),
  plazo_acuerdo: ['plazoacuerdo','plazo acuerdo','plazo_acuerdo','plazo'].map(normKey),
};

function guessField(keysNorm, mapOrig, synonyms) {
  for (const syn of synonyms) {
    if (keysNorm.has(syn)) return mapOrig.get(syn);
  }
  for (const key of keysNorm) {
    if (synonyms.some(s => key.includes(s))) return mapOrig.get(key);
  }
  return null;
}

function extractPhone(value) {
  if (!value) return '';
  const str = String(value).trim();
  if (str.includes('E+') && !isNaN(parseFloat(str))) {
    return parseFloat(str).toFixed(0);
  }
  const cleaned = str.replace(/[^\d+\s]/g, '');
  const phoneMatches = cleaned.match(/[\+]?[\d\s]{7,16}/g);
  if (phoneMatches && phoneMatches.length > 0) {
    return phoneMatches[0].replace(/\s/g, '');
  }
  return '';
}

// ---------------- Robust JSON parse + normalize ----------------
function parseMaybeJson(input) {
  if (input === undefined || input === null) return {};
  if (typeof input === 'object') return input;
  if (typeof input === 'string') {
    try { return JSON.parse(input); } catch (err) {
      try { return JSON.parse(input.trim()); } catch (err2) { return {}; }
    }
  }
  try { return JSON.parse(String(input)); } catch (err) { return {}; }
}

function normalizeMasterConfig(cfg) {
  // convierte { KEY: { value: 'x' } } -> { KEY: 'x' } y deja planos tal cual
  const out = {};
  for (const [k, v] of Object.entries(cfg || {})) {
    if (v && typeof v === 'object' && Object.prototype.hasOwnProperty.call(v, 'value')) {
      out[k] = v.value;
    } else {
      out[k] = v;
    }
  }
  return out;
}

router.get('/', (req, res) => {
  res.json({
    status: 'active',
    message: 'Upload endpoint ready',
    supported_operations: ['POST para subir archivos CSV/Excel'],
    accepted_formats: [
      'text/csv',
      'application/csv', 
      'application/vnd.ms-excel',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    ],
    max_file_size: '15MB',
    required_fields: ['telefono'],
    optional_fields: [
      'identificacion', 'nombrecompleto', 'mejor_perfil', 'origen',
      'dias_mora', 'monto_inicial', 'mod_init_cta', 'referencia',
      'plan', 'fecha_vencimiento', 'tipo_obligacion', 'descuento',
      'min', 'plazo_acuerdo'
    ],
    timestamp: new Date().toISOString()
  });
});


// ---------------- Ruta principal ----------------
router.post('/', upload.single('archivo'), async (req, res) => {
  try {
    // --- Validaciones bsicas del upload ---
    const f = req.file;
    if (!f) return res.status(400).json({ error: 'Archivo no subido' });

    // --- Extraer companyId en varias formas comunes ---
    let companyId = req.body?.companyId || req.body?.company_id || req.body?.company || null;

    // --- Leer config enviado por el cliente (puede venir string JSON o ya objeto) ---
    const rawConfigField = ('config' in req.body) ? req.body.config
                          : ('masterConfig' in req.body) ? req.body.masterConfig
                          : undefined;

    let masterConfig = parseMaybeJson(rawConfigField) || {};

    // Si no vino en config, intentar reconstruir desde campos top-level del body (evita sobrescribir archivo)
    if (!Object.keys(masterConfig).length) {
      const candidate = Object.assign({}, req.body);
      // eliminar campos no relacionados
      delete candidate.archivo;
      delete candidate.file;
      delete candidate.companyId;
      delete candidate.company_id;
      delete candidate.config;
      delete candidate.masterConfig;
      // si hay algo til, lo adoptamos
      if (Object.keys(candidate).length) masterConfig = candidate;
    }

    // Normalizar .value -> valor plano
    masterConfig = normalizeMasterConfig(masterConfig);

    // Re-intentar companyId a partir de la configuracin
    companyId = companyId || masterConfig.companyId || masterConfig.company_id || masterConfig.company || null;

    // Diagnstico (temporal)
    console.log('upload: companyId =', companyId);
    console.log('upload: masterConfig keys =', Object.keys(masterConfig).slice(0, 50));
    if (!Object.keys(masterConfig).length) console.warn('upload: masterConfig vaco (ver cliente)');

    // Crear carpeta destino
    await fsp.mkdir(DEST_DIR, { recursive: true });

    // --- Leer hoja (soporta CSV/Excel porque xlsx puede leer CSV tambin) ---
    const wb = xlsx.readFile(f.path);
    const ws = wb.Sheets[wb.SheetNames[0]];
    const rows = xlsx.utils.sheet_to_json(ws, { defval: '', raw: false });

    const contacts = [];
    let processed = 0, skipped = 0;

    for (const r of rows) {
      const mapOrig = new Map();
      const keysNorm = new Set();

      for (const k of Object.keys(r)) {
        const nk = normKey(k);
        mapOrig.set(nk, k);
        keysNorm.add(nk);
      }

      const detectField = (synonyms) => {
        const detected = guessField(keysNorm, mapOrig, synonyms);
        return detected ? String(r[detected]).trim() : '';
      };

      const telefono = extractPhone(detectField(SYN.telefono));
      if (!telefono) { skipped++; continue; }

      const contacto = {
        telefono: telefono,
        identificacion: detectField(SYN.identificacion),
        nombrecompleto: detectField(SYN.nombrecompleto),
        mejor_perfil: detectField(SYN.mejor_perfil),
        origen: detectField(SYN.origen),
        dias_mora: detectField(SYN.dias_mora),
        monto_inicial: detectField(SYN.monto_inicial),
        mod_init_cta: detectField(SYN.mod_init_cta),
        referencia: detectField(SYN.referencia),
        plan: detectField(SYN.plan),
        fecha_vencimiento: detectField(SYN.fecha_vencimiento),
        tipo_obligacion: detectField(SYN.tipo_obligacion),
        descuento: detectField(SYN.descuento),
        min: detectField(SYN.min),
        plazo_acuerdo: detectField(SYN.plazo_acuerdo),
        company_id: companyId || ''
      };

      contacts.push(contacto);
      processed++;
    }

    // --- Escribir CSV con columnas estandarizadas ---
    const csvHeaders = 'telefono,identificacion,nombrecompleto,mejor_perfil,origen,dias_mora,monto_inicial,mod_init_cta,referencia,plan,fecha_vencimiento,tipo_obligacion,descuento,min,plazo_acuerdo';
    const csvLines = [csvHeaders];

    contacts.forEach(c => {
      csvLines.push([
        c.telefono,
        c.identificacion,
        `"${String(c.nombrecompleto || '').replace(/"/g, '""')}"`,
        c.mejor_perfil,
        c.origen,
        c.dias_mora,
        c.monto_inicial,
        c.mod_init_cta,
        c.referencia,
        c.plan,
        c.fecha_vencimiento,
        c.tipo_obligacion,
        c.descuento,
        c.min,
        c.plazo_acuerdo
      ].join(','));
    });

    await fsp.writeFile(DEST_CSV, csvLines.join('\n'), 'utf8');

    // --- Guardar configuracin y contactos en Redis (TTL 24h) ---
    try {
      await storage.set(`masterConfig:${companyId || 'unknown'}`, masterConfig, 86400);
      await storage.set(`contacts:${companyId || 'unknown'}`, contacts, 86400);
      console.log('? Configuracion guardada en Redis:', {
        companyId: companyId || 'unknown',
        configKeys: Object.keys(masterConfig).length,
        contactsCount: contacts.length
      });
    } catch (err) {
      console.warn('upload: no se pudo guardar en Redis:', err.message);
    }

    // --- Guardar JSON extendido en disco ---
    await fsp.writeFile(DEST_JSON, JSON.stringify({
      metadata: {
        processedAt: new Date().toISOString(),
        sourceFile: f.originalname,
        totalContacts: contacts.length
      },
      masterConfig,
      contacts
    }, null, 2), 'utf8');

    // --- Limpiar temporal ---
    try { await fsp.unlink(f.path); } catch (e) { /* ignore */ }

    // --- Responder ---
    return res.json({
      message: 'Archivo procesado correctamente',
      registros: contacts.length,
      omitidos: skipped,
      companyId: companyId || null
    });

  } catch (err) {
    // limpiar archivo temporal si algo falla
    try { if (req.file?.path) await fsp.unlink(req.file.path); } catch (e) { /* ignore */ }
    console.error('Error procesando upload:', err);
    return res.status(500).json({
      error: 'Error al procesar archivo',
      details: err?.message || String(err)
    });
  }
});

module.exports = router;
