// file: routes/ws/call-handler.js
const { detenerEjecucionLlamadas } = require('../../utils/call-launcher');
const safeClose = require('../../utils/safe-close');
const yardMaster = require('../../utils/yardmaster');
const { WebhookResponse } = require('@jambonz/node-client');
const Adaptor = require('../../utils/ultravox_s2s');
const TranscriptManager = require('../../utils/transcript-manager');
const storage = require('../../utils/storage');
const convertirNumeroATexto = require('../../utils/numero-a-texto');

// ========================= NUEVO: DETECCIN DE BUZN =========================
const detectarBuzonVoz = (transcripcion) => {
  if (!transcripcion) return false;
  
  const palabrasBuzon = [
    'secretaria electronica',
    'secretara electrnica',
    'buzn de voz',
    'buzon de voz',
    'voice mail',
    'mailbox',
    'deje su mensaje',
    'deje un mensaje',
    'after the tone',
    'mensaje despus del tono',
    'leave a message',
    'Servicio de contestador'
  ];
  
  const texto = transcripcion.toLowerCase();
  return palabrasBuzon.some(palabra => texto.includes(palabra));
};

// ========================= NUEVO: NOTIFICACIN DE RESULTADOS =========================
const notificarResultadoLlamada = async (telefono, callSid, resultado, detalles = {}) => {
  try {
    await storage.set(`resultado:${callSid}`, {
      telefono,
      resultado,
      timestamp: new Date().toISOString(),
      detalles
    }, 3600);
    console.log(`?? Resultado notificado: ${telefono} - ${resultado}`);
  } catch (error) {
    console.error('? Error notificando resultado:', error);
  }
};

const service = ({ logger, callService }) => {
  const svc = callService;

  svc.on('session:new', async (session) => {
    const { call_sid } = session;
    
    // Extraer el telfono de la URI SIP (MANTIENE LGICA ORIGINAL)
    const v_to = session.to;
    const v_parts = v_to.split(':')[1].split('@')[0];
    const telefono = v_parts.startsWith('7757') ? v_parts.slice(4) : v_parts;
    
    session.locals = {
      logger: logger.child({ call_sid }),
      call_sid,
      telefono,
      // NUEVO: Seguimiento de estado
      startTime: new Date(),
      resultado: 'en_progreso'
    };
    
    try {
      // MANTIENE LGICA ORIGINAL DE DATOS
      const datos = await storage.get(`callinfo:${telefono}`);
      if (!datos) {
        session.locals.logger.error('No se encontraron datos del contacto en Redis');
        // NUEVO: Notificar fallo
        await notificarResultadoLlamada(telefono, call_sid, 'fallido', { error: 'datos_no_encontrados' });
        return session.hangup().send();
      }

      session.locals.identificacion = datos.identificacion;
      session.locals.datosContacto = datos;

      // MANTIENE LGICA ORIGINAL DE API KEY
      const apiKey = process.env.ULTRAVOX_API_KEY;
      if (!apiKey) {
        session.locals.logger.error('ULTRAVOX_API_KEY missing, hanging up');
        // NUEVO: Notificar fallo
        await notificarResultadoLlamada(telefono, call_sid, 'fallido', { error: 'api_key_missing' });
        return session.hangup().send();
      }

      session.locals.logger.info({ session, datos }, `[CALL] Nueva llamada entrante: ${call_sid}`);
      yardMaster.addSession(call_sid);
      
      // MANTIENE FUNCIONES ORIGINALES
      const parseDecimal = (str) => {
        if (typeof str !== 'string') return str;
        return parseFloat(str.replace(/,/g, '').replace(/\s+/g, '').trim());
      };

      const pronunciarDigitos = (numeroStr) => {
        return String(numeroStr).split('').join(', ');
      };

      // ========================= NUEVO: PROMPT MEJORADO =========================
      const buildSystemPrompt = (datosContacto) => {
        const cliente = datosContacto.nombrecompleto || 'Estimado cliente';
        //const cuenta = datosContacto.cuenta || 'XXXXXXXX';
        //const cuentaEnDigitos = cuenta.split('').join(', ');
        const monto = parseDecimal(datosContacto.monto_inicial) || 0;
        const montoEnLetras = convertirNumeroATexto(monto);
        
        // NUEVO: Mapeo de tipos de servicio (usa campo origen del CSV)
        const mapearTipoServicio = (origen) => {
          const mapeo = {
            'BSCS': 'Servicio Mvil',
            'ASCARD': 'Servicio Hogar', 
            'RR': 'Equipo Financiado'
          };
          return mapeo[origen] || origen || 'No especificado';
        };

        const tipoServicio = mapearTipoServicio(datosContacto.origen);
        const plazoAcuerdo = datosContacto.plazo_acuerdo || 'prximos 5 das hbiles';

        return `
      ROLE: Eres Andrea, agente especializada en cobranza de Claro. Tu estilo es:
      - Emptico pero profesional y directa
      - Clara en la comunicacin
      - Persuasiva pero no agresiva
      - Orientada a soluciones
      - Buena pronunciacin de nmeros
      
      CONTEXTO:
      Ests contactando a ${cliente} de parte de "ATM BPO Aliado estratgico para Claro". 
      Le informa que esta llamada est siendo grabada y monitoreada.
      
      DETALLE DE LA OBLIGACIN:
      - Producto: ${tipoServicio} (${datosContacto.tipo_obligacion || 'No especificado'})
      - Nmero de cuenta: ${pronunciarDigitos(datosContacto.referencia)}
      - Valor adeudado: $${montoEnLetras}
      - Fecha de vencimiento: ${datosContacto.fecha_vencimiento || 'No especificada'}
      - Das en mora: ${datosContacto.dias_mora || '0'} das
      - Fecha mxima acuerdo: ${plazoAcuerdo}
      - Plan: ${datosContacto.plan || 'No especificado'}
      - Referencia: ${datosContacto.referencia || 'No especificado'}
      
      INSTRUCCIONES MEJORADAS:
      1. Saluda cordialmente: "Buenos das/tardes, habla Andrea de ATM BPO, aliado estratgico para Claro"
      2. Informa: "Le informo que esta llamada est siendo grabada y monitoreada"
      3. Confirma que hablas con ${cliente.split(' ')[0] || 'el cliente'}, "si no lo es pregunta si el encargado de los pagos"
      4. Menciona el motivo: "Le llamo acerca de su obligacin vencida con Claro por el ${tipoServicio.toLowerCase()}"
      5. Informa el monto adeudado: $${montoEnLetras}
      6. Menciona los das de mora: ${datosContacto.dias_mora} das
      7. Ofrece alternativas de pago (pago total)
      8. Si cliente dice "nmero equivocado": confirmar y terminar cordialmente
      9. Si cliente indica "ya pag": preguntar detalles y terminar
      10. Si detectas buzn de voz: "terminar inmediatamente con las frases Servicio de contestador"
      11. Los pagos se realizan solamente por el app o la web de Claro
      12. Mantn la conversacin fluida y profesional
      13. Cuando leas nmeros, usa pausas naturales
      14. Si el cliente pide referencia: informar ${datosContacto.referencia}
      
      CONSECUENCIAS SEGN SERVICIO:
      - Servicio Mvil: "Suspensin del servicio y afectacin crediticia"
      - Servicio Hogar: "Intereses por mora y suspensin del servicio"
      - Equipo Financiado: "Afectacin de capacidad de financiamiento"
      `.trim();
      };
      // ========================= FIN PROMPT MEJORADO =========================
            
      const fullSystemPrompt = buildSystemPrompt(datos);

      // MANTIENE ADAPTADOR ORIGINAL
      const adaptor = new Adaptor(session.locals.logger, apiKey);
      session.locals.adaptor = adaptor;
     
      adaptor.on('ready', () => {
        session.locals.logger.info(' Adaptador Ultravox listo (ready)');
        adaptor.setOutgoingJambonzSocket(session.ws);
      });
      
      adaptor.setIncomingJambonzSocket(session.ws);
 
      // MANTIENE EVENTOS ORIGINALES
      session
        .on('/event', onEvent.bind(null, session))
        .on('/toolCall', onToolCall.bind(null, session))
        .on('/final', onFinal.bind(null, session))
        .on('/sip_referAction', sipReferAction.bind(null, session))
        .on('/sip_referEvent', sipReferEvent.bind(null, session))
        .on('call:status', onCallStatus.bind(null, session))
        .on('close', onClose.bind(null, session))
        .on('error', onError.bind(null, session));

      // MANTIENE CONFIGURACIN ORIGINAL
      session.config({
        listen: {
          enable: true,
          url: `${process.env.WS_BASE_URL}/audio-stream?callSid=${call_sid}`,
          mixType: 'mono',
          sampleRate: 8000,
          bidirectionalAudio: {
            enabled: true,
            streaming: true,
            sampleRate: 8000
          }
        }
      });

      // MANTIENE LLM ORIGINAL
      session.llm({
        vendor: 'ultravox',
        model: 'fixie-ai/ultravox',
        auth: { apiKey },
        actionHook: '/final',
        eventHook: '/event',
        toolHook: '/toolCall',
        llmOptions: {
          systemPrompt: fullSystemPrompt,
          firstSpeaker: 'FIRST_SPEAKER_AGENT',
          initialMessages: [
            {
              medium: 'MESSAGE_MEDIUM_VOICE',
              role: 'MESSAGE_ROLE_AGENT',
              text: 'Hola, soy del rea de cobranza.',
            }
          ],
          model: 'fixie-ai/ultravox',
          voice: 'Andrea-Spanish',
          transcriptOptional: true,
        }
      }).send();
     
    } catch (err) {
      session.locals.logger.error({ err }, 'Error en session:new');
      // NUEVO: Notificar error
      await notificarResultadoLlamada(telefono, call_sid, 'fallido', { error: err.message });
      safeClose(session.locals.logger, session.call_sid);
      session.close();
    }
  });
};

// ========================= EVENTOS MEJORADOS =========================
const onEvent = async (session, evt) => {
  const { logger, call_sid, telefono } = session.locals;
  logger.info({ evt }, 'Evento recibido en /event');

  // MANTIENE LGICA ORIGINAL DE TRANSCRIPCIN
  if (evt?.type === 'transcript' && evt?.final) {
    const turns = Array.isArray(evt.transcript)
      ? evt.transcript
      : [{ role: evt.role, text: evt.text ?? evt.message }];

    try {
      if (!session.locals.call_sid) {
        logger.warn({ evtKeys: Object.keys(evt || {}) }, 'transcript.final sin call_sid');
        return session.reply();
      }

      for (const t of turns) {
        const message = (t?.text ?? t?.message ?? '').trim();
        if (!message) continue;

        // ========================= NUEVO: DETECCIN DE BUZN EN TIEMPO REAL =========================
        if (t.role === 'customer' && detectarBuzonVoz(message)) {
          logger.info(`?? Buzn detectado para ${telefono}, colgando llamada`);
          session.locals.resultado = 'buzon';
          
          await notificarResultadoLlamada(telefono, call_sid, 'buzon', {
            mensaje: message,
            accion: 'terminacion_automatica'
          });
          
          setTimeout(() => {
            session.hangup().send();
          }, 1000);
          break;
        }
        // ========================= FIN DETECCIN BUZN =========================
        
        // MANTIENE LGICA ORIGINAL DE GUARDADO
        await TranscriptManager.saveMessage(session.locals.call_sid, {
          speaker: (t?.role || 'USER').toUpperCase(),
          message
        });
      }
    } catch (err) {
      logger.error({ err, evtShape: Object.keys(evt || {}) }, 'Error guardando transcripcin completa');
    }
  }

  session.reply();
};

// MANTIENE toolCall ORIGINAL
const onToolCall = async (session, evt) => {
  const { logger, telefono, call_sid } = session.locals;
  const { tool_call_id } = evt;
  logger.info({ evt }, ' Tool hook recibido');

  try {
    // NUEVO: Manejo de herramientas especiales
    if (evt.name === 'terminar_llamada_equivocada') {
      logger.info(`?? Terminando por nmero equivocado: ${telefono}`);
      session.locals.resultado = 'equivocado';
      await notificarResultadoLlamada(telefono, call_sid, 'equivocado');
      
      setTimeout(() => {
        session.hangup().send();
      }, 2000);
    }

    // MANTIENE RESPUESTA ORIGINAL
    const data = {
      type: 'client_tool_result',
      invocation_id: tool_call_id,
      result: 'Operacin realizada correctamente',
    };
    session.sendToolOutput(tool_call_id, data);
  } catch (err) {
    logger.error({ err }, 'Error enviando resultado de herramienta');
  }
};

// ========================= onFinal MEJORADO =========================
const onFinal = async (session, evt) => {
  const { logger, telefono, call_sid, resultado } = session.locals;
  logger.info({ evt }, 'Evento final recibido');
  
  // MANTIENE LGICA ORIGINAL DE TRANSCRIPCIN
  if (evt.transcript && Array.isArray(evt.transcript)) {
    try {
      for (const turn of evt.transcript) {
        await TranscriptManager.saveMessage(session.call_sid, {
          speaker: turn.role === 'agent' ? 'AGENT' : 
                  turn.role === 'customer' ? 'CUSTOMER' : 'BOT',
          message: turn.text || turn.message || '',
          timestamp: new Date(turn.timestamp || Date.now())
        });
      }
      logger.info(`Guardada transcripcin final con ${evt.transcript.length} mensajes`);
    } catch (error) {
      logger.error({ error }, 'Error guardando transcripcin final');
    }
  }

  // NUEVO: Notificar resultado final
  const resultadoFinal = resultado === 'en_progreso' ? 'exitoso' : resultado;
  await notificarResultadoLlamada(telefono, call_sid, resultadoFinal, {
    duracion: Date.now() - session.locals.startTime
  });

  session.reply();
};

// MANTIENE FUNCIONES ORIGINALES SIN MODIFICACIONES
const onCallStatus = (session, evt) => {
  const { logger, telefono, call_sid } = session.locals;
  logger.info({ evt }, 'Estado de llamada');
  
  // NUEVO: Notificar estados de fallo
  if (evt.call_status === 'failed' || evt.call_status === 'no-answer') {
    notificarResultadoLlamada(telefono, call_sid, 'fallido', {
      estado: evt.call_status,
      motivo: evt.reason || 'desconocido'
    });
  }
  
  if (!session.locals.call_sid_b && evt.direction === 'outbound') {
    session.locals.call_sid_b = evt.call_sid;
  }
};

const sipReferAction = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, '[SIP] REFER Action');
};

const sipReferEvent = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, '[SIP] REFER Event');
};

// ========================= onClose MEJORADO =========================
const onClose = (session, code, reason) => {
  const { logger, telefono, call_sid, resultado } = session.locals;
  logger.info({ code, reason }, ' Sesin cerrada');
  
  // NUEVO: Notificar cierre si no se notific resultado
  if (resultado === 'en_progreso') {
    const resultadoCierre = code === 1000 ? 'exitoso' : 'fallido';
    notificarResultadoLlamada(telefono, call_sid, resultadoCierre, {
      codigo_cierre: code,
      razon: reason
    });
  }
  
  // MANTIENE LGICA ORIGINAL
  detenerEjecucionLlamadas();
  safeClose(logger, session.call_sid);
};

// ========================= onError MEJORADO =========================
const onError = (session, err) => {
  const { logger, telefono, call_sid } = session.locals;
  logger.error({ err }, 'Error en sesin');
  
  // NUEVO: Notificar error
  notificarResultadoLlamada(telefono, call_sid, 'fallido', {
    error: err.message,
    tipo: 'error_sesion'
  });
  
  // MANTIENE LGICA ORIGINAL
  safeClose(logger, session.call_sid);
};

module.exports = service;