const cron = require('node-cron');
const { ejecutarLlamadas } = require('../utils/call-launcher');
const logger = require('../utils/logger')('bot-scheduler');
const fs = require('fs');
const path = require('path');
const { DateTime } = require('luxon');
const { CronExpressionParser } = require('cron-parser');

const CONFIG_PATH = path.join(__dirname, '../config/horario-atm.json');
let ejecucionEnCurso = false;
const interval = CronExpressionParser.parse('*/5 * * * *');
// Helper para obtener contexto de ejecución
const getContext = (depth = 2) => {
  const stackTrace = new Error().stack.split('\n');
  const callerLine = stackTrace[depth]?.match(/at (.+) \((.+):(\d+):(\d+)\)/) || 
                    stackTrace[depth]?.match(/at (.+):(\d+):(\d+)/);
  
  return {
    method: callerLine?.[1]?.trim() || 'anonymous',
    file: callerLine?.[2] ? path.basename(callerLine[2]) : 'unknown',
    line: callerLine?.[3] || '0',
    timestamp: new Date().toISOString()
  };
};

function cargarConfiguracionHorario() {
  const context = getContext();
  try {
    const data = fs.readFileSync(CONFIG_PATH, 'utf8');
    logger.debug('Configuración de horario cargada exitosamente', context);
    return JSON.parse(data);
  } catch (err) {
    logger.error('Error al leer archivo de configuración de horario', {
      ...context,
      error: {
        message: err.message,
        code: err.code,
        path: CONFIG_PATH
      }
    });
    return null;
  }
}

function estaDentroDeHorario() {
  const context = getContext();
  const config = cargarConfiguracionHorario();
  
  if (!config) {
    logger.warn('No se pudo cargar la configuración de horario', context);
    return false;
  }

  try {
    const zona = config.timezone || 'America/Bogota';
    const ahora = DateTime.now().setZone(zona);
    const diasSemana = ['domingo', 'lunes', 'martes', 'miercoles', 'jueves', 'viernes', 'sabado'];
    const hoy = diasSemana[ahora.weekday % 7];
    const diaConfig = config.dias[hoy];

    if (!diaConfig || !diaConfig.activo) {
      logger.debug(`Día ${hoy} inactivo en configuración`, context);
      return false;
    }

    const [hIni, mIni] = diaConfig.inicio.split(':').map(Number);
    const [hFin, mFin] = diaConfig.fin.split(':').map(Number);
    const horaInicio = ahora.set({ hour: hIni, minute: mIni, second: 0 });
    const horaFin = ahora.set({ hour: hFin, minute: mFin, second: 0 });
    const dentroDeHorario = ahora >= horaInicio && ahora < horaFin;

    logger.debug(`Verificación de horario para ${hoy}: ${dentroDeHorario ? 'DENTRO' : 'FUERA'} del horario`, {
      ...context,
      detalles: {
        horaActual: ahora.toISO(),
        horarioConfig: `${diaConfig.inicio} - ${diaConfig.fin}`,
        zonaHoraria: zona
      }
    });

    return dentroDeHorario;
  } catch (err) {
    logger.error('Error al verificar horario', {
      ...context,
      error: {
        message: err.message,
        stack: err.stack
      }
    });
    return false;
  }
}

function iniciarScheduler() {
  const context = getContext();
  
  try {
    const job = cron.schedule('*/5 * * * *', async () => {
      const executionContext = {
        ...getContext(3),
        executionId: Date.now()
      };

      try {
        if (!estaDentroDeHorario()) {
          logger.info('Fuera del horario programado para llamadas ATM', executionContext);
          return;
        }

        if (ejecucionEnCurso) {
          logger.warn('Ejecución omitida: ya hay una ejecución en curso', executionContext);
          return;
        }

        logger.info('Iniciando ejecución programada de llamadas ATM', executionContext);
        ejecucionEnCurso = true;
        
        await ejecutarLlamadas();
        
        logger.info('Ejecución de llamadas ATM completada con éxito', {
          ...executionContext,
          status: 'completed'
        });
      } catch (err) {
        logger.error('Error durante la ejecución programada', {
          ...executionContext,
          error: {
            message: err.message,
            stack: err.stack,
            code: err.code
          },
          status: 'failed'
        });
      } finally {
        ejecucionEnCurso = false;
      }
    });

    job.start();
    logger.info('Scheduler ATM dinámico activo cada 5 minutos', {
      ...context,
      config: {
        pattern: '*/5 * * * *',
        timezone: 'America/Bogota',
        //firstExecution: job.nextDates().toISO()
        firstExecution: interval.next().toISOString()
      }
    });

    return job;
  } catch (err) {
    logger.error('Error al iniciar el scheduler', {
      ...context,
      error: {
        message: err.message,
        stack: err.stack
      }
    });
    throw err;
  }
}

module.exports = {
  iniciarScheduler,
  getContext // Exportado para testing
};