/**
 * Archivo: atm-service.js
 * Mejoras:
 * - Detección automática de perfil desde texto.
 * - Detección automática de motivo de no pago desde texto.
 */

const axios = require('axios');
const https = require('https');
const fs = require('fs').promises;
const path = require('path');
const { setTimeout } = require('timers/promises');
const logger = require('./../utils/logger')('atm-service');
const crypto = require('crypto'); 

const MAX_RETRIES = 10;
const BASE_DELAY_MS = 30000;
const ATM_BASE_URL = 'https://atmbposas.controlnextapp.com:7050';
const ENDPOINTS = {
  AUTH: `${ATM_BASE_URL}/auth/login`,
  USERS: `${ATM_BASE_URL}/users`,
  GESTION: `${ATM_BASE_URL}/users/gestion`
};

const httpsAgent = new https.Agent({
  keepAlive: true,
  maxSockets: 150,
  timeout: 30000,
  rejectUnauthorized: true,
  secureOptions: crypto.constants.SSL_OP_LEGACY_SERVER_CONNECT
});

const tokenManager = (() => {
  let currentToken = null;
  let expiry = null;
  let refreshPromise = null;

  return {
    async getToken() {
      if (currentToken && Date.now() < expiry) return currentToken;
      if (!refreshPromise) {
        refreshPromise = this.refreshToken().finally(() => refreshPromise = null);
      }
      return refreshPromise;
    },

 async refreshToken() {
  try {
    const credentials = {
      username: process.env.ATM_USER,
      password: process.env.ATM_PASS
    };

    logger.debug('Intentando autenticar con ATM', {
      url: ENDPOINTS.AUTH,
      username: credentials.username
    });

    const response = await this.retryableRequest({
      method: 'post',
      url: ENDPOINTS.AUTH,
      data: credentials,
      timeout: 8000
    });

    if (!response.data.access_token) {
      throw new Error('No se recibió token en la respuesta');
    }

    logger.debug('Autenticación exitosa con ATM');

    currentToken = response.data.access_token;
    expiry = Date.now() + (23 * 60 * 60 * 1000); // 23 horas

    console.log(" Token activo:", currentToken);
    console.log(" Expira en:", new Date(expiry).toISOString());

    return currentToken;
  } catch (error) {
    logger.error('Error en refreshToken', {
      error: error.message,
      response: error.response?.data,
      config: error.config
    });
    throw error;
  }
}
,

    async retryableRequest(config, retries = MAX_RETRIES) {
      let lastError;
      for (let attempt = 1; attempt <= retries; attempt++) {
        try {
          return await axios({ httpsAgent, ...config });
        } catch (error) {
          lastError = error;
          if (error.response?.status === 401 && attempt === 1) {
            currentToken = null;
            continue;
          }
          if (attempt < retries) await setTimeout(BASE_DELAY_MS * 2 ** (attempt - 1));
        }
      }
      throw lastError;
    }
  };
})();

function detectarAcuerdo(texto) {
  const patrones = [/acepto/i, /har[ée]/i, /comprometo/i, /promesa/i, /acuerdo/i, /cuota/i, /sí voy a pagar/i];
  return patrones.some(r => r.test(texto));
}

function detectarFechaPromesa(texto) {
  const iso = texto.match(/\d{4}-\d{2}-\d{2}/);
  if (iso) return iso[0];
  const natural = texto.match(/(0?[1-9]|[12][0-9]|3[01])\s+(de\s+)?(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i);
  if (natural) {
    const [_, dia, __, mes] = natural;
    const meses = { enero: '01', febrero: '02', marzo: '03', abril: '04', mayo: '05', junio: '06', julio: '07', agosto: '08', septiembre: '09', octubre: '10', noviembre: '11', diciembre: '12' };
    return `${new Date().getFullYear()}-${meses[mes.toLowerCase()]}-${String(dia).padStart(2, '0')}`;
  }
  
    // Extensión: "hoy" o "mañana"
  if (/\bhoy\b/i.test(texto)) {
    const d = new Date(); return d.toISOString().slice(0,10);
  }
  if (/\bmañana\b/i.test(texto)) {
    const d = new Date(); d.setDate(d.getDate()+1); return d.toISOString().slice(0,10);
  }

  return null;
}

function detectarPerfilDesdeTexto(texto) {
  const perfilMap = [
    { regex: /promesa|comprometo|har[ée]/i, valor: 10 },
    { regex: /volver.*llamar|llama(r)? (luego|más tarde)/i, valor: 23 },
    { regex: /no quiero pagar|renuente/i, valor: 4 },
    { regex: /ya pag[ée]/i, valor: 5 },
    { regex: /no puedo pagar|dificultad/i, valor: 2 },
    { regex: /recordar/i, valor: 7 },
    { regex: /colg[óo]/i, valor: 15 },
    { regex: /pagar[ée]?/i, valor: 24 }
  ];
  for (const { regex, valor } of perfilMap) if (regex.test(texto)) return valor;
  return 24;
}

function detectarMotivoNoPago(texto) {
  const motivos = [
    { regex: /sin (trabajo|ingresos|plata)/i, valor: 41 },
    { regex: /no reconoce/i, valor: 56 },
    { regex: /final de mes/i, valor: 57 },
    { regex: /inconformidad.*servicio/i, valor: 42 },
    { regex: /aumento.*tarifa/i, valor: 43 },
    { regex: /PQR/i, valor: 44 },
    { regex: /calamidad/i, valor: 45 },
    { regex: /equipo/i, valor: 46 },
    { regex: /encargado.*pago/i, valor: 47 },
    { regex: /no confirma/i, valor: 48 },
    { regex: /viaje/i, valor: 49 },
    { regex: /no recib.*factura/i, valor: 50 },
    { regex: /prioridades.*pago/i, valor: 51 },
    { regex: /salud|m[eé]dico/i, valor: 52 },
    { regex: /inconsistencia.*(pago|puntos)/i, valor: 53 },
    { regex: /cancelaci[oó]n.*no atendida/i, valor: 54 }
  ];
  for (const { regex, valor } of motivos) if (regex.test(texto)) return valor;
  return undefined;
}

/*function extraerResumenDesdeTranscripcion(transcripts = []) {
  return transcripts.filter(t => t.role === 'agent' && t.final).map(t => t.text).filter(Boolean).join(' ');
}
*/

// Preferimos lo que dijo el CLIENTE; si no hay, caemos al agente
function extraerResumenUsuario(transcripts = []) {
  const userRoles = new Set(['user', 'caller']);
  const userText = transcripts
    .filter(t => t.final && userRoles.has(t.role))
    .map(t => t.text)
    .filter(Boolean)
    .join(' ');
  return userText;
}
function extraerResumenAgente(transcripts = []) {
  return transcripts
    .filter(t => t.final && t.role === 'agent')
    .map(t => t.text)
    .filter(Boolean)
    .join(' ');
}
function extraerResumenDesdeTranscripcion(transcripts = []) {
  return extraerResumenUsuario(transcripts) || extraerResumenAgente(transcripts) || '';
}

function confirmarAcuerdo(fecha, valor, referencia) {
  return `Tiene un acuerdo de pago por $${valor} con fecha ${fecha} y referencia ${referencia}.`;
}



function normalizarMonto(rawValue) {
  // Elimina cualquier carácter que no sea número o punto
  const limpio = String(rawValue).replace(/[^\d.]/g, '');

  // Si tiene más de un punto decimal, asume que es separador de miles mal usado (como 85.653.00) → quítalos todos menos el último
  const partes = limpio.split('.');
  if (partes.length > 2) {
    const parteDecimal = partes.pop(); // última parte
    const parteEntera = partes.join('');
    return parseFloat(`${parteEntera}.${parteDecimal}`);
  }

  return parseFloat(limpio);
}


function construirPayloadATM({ telefono, identificacion, duracionSegundos = 0, transcripciones = [], cuentas = [], codllamada, codigo_gestion }) {
  // 1) Resumen: prioriza lo que dijo el CLIENTE; si no hay, usa lo del agente
  const resumenUsuario = extraerResumenUsuario(transcripciones);
  const resumenAgente  = extraerResumenAgente(transcripciones);
  const resumen        = resumenUsuario || resumenAgente || '';

  // 2) Clasificación
  let perfil          = detectarPerfilDesdeTexto(resumen);
  const motivo        = detectarMotivoNoPago(resumen);
  const huboAcuerdo   = detectarAcuerdo(resumenUsuario);      // acuerdo solo si lo dijo el CLIENTE
  const fechaPromesa  = detectarFechaPromesa(resumenUsuario); // fecha desde lo que dijo el CLIENTE

  // 3) Payload base
  const payload = {
    perfil,
    gestion: resumen || 'Cliente no respondió con claridad',
    telefono: String(telefono),
    identificacion: String(identificacion),
    segundostotales: Number.isFinite(Number(duracionSegundos))
      ? Math.max(0, Math.floor(Number(duracionSegundos)))
      : 0,
    accion: 1,
    tipocontacto: 1
  };

  // 4) Acuerdo (si hubo y hay datos válidos)
  if (huboAcuerdo && cuentas.length > 0) {
    const cuentaValida = typeof cuentas[0].cuenta === 'string' && cuentas[0].cuenta.trim();
    const monto = normalizarMonto(cuentas[0].mod_init_cta);
    if (fechaPromesa && cuentaValida && !isNaN(monto)) {
      Object.assign(payload, {
        cuenta: [cuentas[0].cuenta],
        valor_cuota: [monto],
        valor_total: [monto],
        fechapromesa: [fechaPromesa]
      });
      // Con acuerdo válido, fija perfil a Promesa (10)
      payload.perfil = 10;
    } else {
      logger.warn('No se enviará acuerdo porque los datos requeridos están incompletos');
    }
  }

  // 5) Motivo de no pago (no enviar si hubo acuerdo)
  if (huboAcuerdo) {
    delete payload.motivonopago;
  } else {
    if ([2, 23, 15].includes(payload.perfil)) {
      delete payload.motivonopago;
    } else if (typeof motivo === 'number') {
      payload.motivonopago = motivo;
    } else {
      payload.motivonopago = 48; // fallback "no confirma"
    }
  }

  // 6) Limpieza cuando NO hubo acuerdo
  if (!huboAcuerdo) {
    delete payload.valor_cuota;
    delete payload.valor_total;
    delete payload.fechapromesa;
    if ([2, 23, 15].includes(payload.perfil)) delete payload.cuenta;
  }

  // 7) Extras
  if (codllamada)    payload.codllamada    = codllamada;
  if (codigo_gestion) payload.codigo_gestion = codigo_gestion;

  return payload;
}


async function loadCertificates() {
  const [ca, cert, key] = await Promise.all([
    fs.readFile(path.join(__dirname, '../certs/ca_bundle.crt'), 'utf8'),
    fs.readFile(path.join(__dirname, '../certs/certificate.crt'), 'utf8'),
    fs.readFile(path.join(__dirname, '../certs/private.key'), 'utf8').catch(() => null)
  ]);
  Object.assign(httpsAgent.options, { ca, cert });
  if (key) httpsAgent.options.key = key;
}

module.exports = {
  httpsAgent,
  loadCertificates,
  construirPayloadATM,
  async obtenerTokenATM() {
    return await tokenManager.getToken();
  },

  async consultarObligaciones(identificacion) {
  try {
    console.log("ingresa a obligaciones");
    const idNum = typeof identificacion === 'string' ? parseInt(identificacion, 10) : identificacion;
    const token = await tokenManager.getToken();
    
    logger.debug('Consultando obligaciones', {
      identificacion: idNum,
      token: token ? '***' : 'No disponible'
    });

    const response = await tokenManager.retryableRequest({
      method: 'post',
      url: ENDPOINTS.USERS,
      data: { identificacion: idNum },
      headers: {
        Authorization: `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (!response.data) {
      throw new Error('Respuesta vacía de ATM');
    }

    logger.debug('Respuesta de obligaciones recibida', {
      dataLength: Array.isArray(response.data) ? response.data.length : 1
    });

    return response.data;
  } catch (error) {
    logger.error('Error en consultarObligaciones', {
      error: error.message,
      response: error.response?.data,
      stack: error.stack
    });
    throw error;
  }
},

  async enviarResultadoATM(payload) {
    const token = await tokenManager.getToken();
    const response = await tokenManager.retryableRequest({
      method: 'post',
      url: ENDPOINTS.GESTION,
      data: payload,
      headers: {
        Authorization: `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      timeout: 15000
    });
    return response.data;
  },

 _test: {
    detectarAcuerdo,
    detectarFechaPromesa,
    detectarPerfilDesdeTexto,
    detectarMotivoNoPago,
    extraerResumenDesdeTranscripcion,
    extraerResumenUsuario,
    extraerResumenAgente,
    construirPayloadATM
  }
};
