// utils/caller.js
require('dotenv').config();
const fs = require('fs');
const fsp = fs.promises;
const path = require('path');
const csv = require('csv-parser');
const jambonz = require('@jambonz/node-client');

const storage = require('./storage');

// ========================= CONSTANTES ORIGINALES =========================
const CONTACTS_FILE  = path.join(__dirname, '../data/contactos.csv');
const PROCESSED_FILE = path.join(__dirname, '../data/contactos_procesados.json');
const BATCH_SIZE = 5;

// ========================= NUEVAS CONSTANTES =========================
const MAX_REINTENTOS = 3; // Mximo de reintentos por nmero

let detenerLlamadas = false;
let llamadasActivas = 0;
let lotesCompletados = 0;
let masterConfig = {};
let currentCompanyId = null;

// ========================= CONFIGURACIN DESDE REDIS =========================
async function cargarConfiguracionDesdeRedis(companyId = null) {
  try {
    // Si no se pasa companyId, intentar obtener del primer contacto
    if (!companyId) {
      const contacts = await obtenerContactos();
      if (contacts.length > 0 && contacts[0].company_id) {
        companyId = contacts[0].company_id;
      }
    }
    
    if (companyId) {
      currentCompanyId = companyId;
      let cfg = await storage.get(`masterConfig:${companyId}`) || {};
      // Si cfg es string JSON, parsearlo
      if (typeof cfg === 'string') {
        try { cfg = JSON.parse(cfg); } catch (_) { cfg = {}; }
      }
      masterConfig = normalizeMasterConfig(cfg);
      
      console.log('? Configuracin cargada desde Redis:', {
        companyId,
        configKeys: Object.keys(masterConfig).length,
        tieneCALLER_ID: !!getCfg('CALLER_ID_OVERRIDE'),
        tieneAPPLICATION_SID: !!getCfg('APPLICATION_SID')
      });
    } else {
      console.log('?? No se pudo determinar companyId para cargar configuracin');
      masterConfig = {};
    }
  } catch (error) {
    console.error('? Error cargando configuracin desde Redis:', error);
    masterConfig = {};
  }
}

// ========================= OBTENER CONTACTOS DESDE REDIS O CSV =========================
async function obtenerContactos() {
  try {
    // Primero intentar desde Redis (si companyId est presente)
    if (currentCompanyId) {
      const contacts = await storage.get(`contacts:${currentCompanyId}`);
        console.log("ver si hay contactos:", contacts );
      if (contacts && contacts.length > 0) {
        console.log(`? Contactos cargados desde Redis: ${contacts.length}`);
        return contacts;
      }
    }
    
    // Fallback: leer desde CSV
    return new Promise((resolve, reject) => {
      const contacts = [];
      fs.createReadStream(CONTACTS_FILE)
        .pipe(csv())
        .on('data', (row) => {
          if (row.telefono && String(row.telefono).trim() !== '') {
            contacts.push(row);
          }
        })
        .on('end', () => {
          console.log(`? Contactos cargados desde CSV: ${contacts.length}`);
          resolve(contacts);
        })
        .on('error', reject);
    });
  } catch (error) {
    console.error('? Error obteniendo contactos:', error);
    return [];
  }
}

// ========================= HELPERS DE CONFIG =========================
function normalizeMasterConfig(cfg) {
  // convierte { KEY: { value: 'x' } } -> { KEY: 'x' } y deja planos tal cual
  const out = {};
  if (!cfg || typeof cfg !== 'object') return out;
  for (const [k, v] of Object.entries(cfg)) {
    if (v && typeof v === 'object' && Object.prototype.hasOwnProperty.call(v, 'value')) {
      out[k] = v.value;
    } else {
      out[k] = v;
    }
  }
  return out;
}

// obtiene valor de la config probando X, X.value y process.env fallbacks
function getCfg(key, envFallback = null) {
  if (!masterConfig) return envFallback ? process.env[envFallback] : undefined;
  if (Object.prototype.hasOwnProperty.call(masterConfig, key)) {
    return masterConfig[key];
  }
  // try upper/lower variants
  if (masterConfig[key.toUpperCase()]) return masterConfig[key.toUpperCase()];
  if (masterConfig[key.toLowerCase()]) return masterConfig[key.toLowerCase()];
  // finally fallback to env var name if provided
  if (envFallback) return process.env[envFallback];
  return undefined;
}

// ========================= CLIENT JAMBONZ DINMICO =========================
function getJambonzClient() {
  // Intentar leer accountSid y apiKey tanto del masterConfig plano como de env vars
  const accountSid = getCfg('JAMBONZ_ACCOUNT_SID', 'JAMBONZ_ACCOUNT_SID') || getCfg('JAMBONZ_ACCOUNT', 'JAMBONZ_ACCOUNT_SID');
  const apiKey = getCfg('JAMBONZ_API_KEY', 'JAMBONZ_API_KEY') || getCfg('JAMBONZ_API', 'JAMBONZ_API_KEY');
  const baseUrl = getCfg('JAMBONZ_API_URL', 'JAMBONZ_BASE_URL') || getCfg('JAMBONZ_BASE_URL', 'JAMBONZ_BASE_URL') || process.env.JAMBONZ_URL;

  if (!accountSid || !apiKey) {
    // lanzar error para que el caller lo capture y lo marque en estado
    throw new Error('Faltan credenciales de Jambonz (accountSid o apiKey).');
  }

  try {
    return jambonz(accountSid, apiKey, { baseUrl });
  } catch (error) {
    console.error('? Error creando cliente Jambonz:', error);
    throw error;
  }
}

// ========================= HISTORIAL ORIGINAL =========================
let numerosProcesados = {
  ultimaEjecucion: null,
  numerosLlamados: new Set(),          // Set<string>
  intentos: new Map()                  // Map<string, number>
};

// ========================= NUEVO: ESTADO MEJORADO =========================
let estadoMejorado = {
  errores: new Map(),                  // Map<string, {status: number, count: number, lastAttempt: Date}>
  exitosos: new Set(),                 // Set<string> - nmeros que terminaron exitosamente
  resultados: new Map()                // Map<callSid, {telefono, resultado, detalles}>
};

// Esperas por finalizacin de llamadas (ORIGINAL)
const PENDING   = new Map(); // callSid -> resolver()
const COMPLETED = new Set(); // callSid que termin antes de que armemos el waiter

// ========================= NUEVAS FUNCIONES AUXILIARES =========================
function marcarExitoso(telefono) {
  estadoMejorado.exitosos.add(telefono);
  estadoMejorado.errores.delete(telefono);
}

function marcarError(telefono, status) {
  const errorInfo = estadoMejorado.errores.get(telefono) || { 
    status: 0, 
    count: 0, 
    lastAttempt: null 
  };
  errorInfo.status = status;
  errorInfo.count += 1;
  errorInfo.lastAttempt = new Date();
  estadoMejorado.errores.set(telefono, errorInfo);
}

function puedeReintentar(telefono) {
  const errorInfo = estadoMejorado.errores.get(telefono);
  if (!errorInfo) return true;
  return errorInfo.count < MAX_REINTENTOS;
}

function yaFueExitoso(telefono) {
  return estadoMejorado.exitosos.has(telefono);
}

// ========================= NUEVO: CONSULTA DE RESULTADOS =========================
async function consultarResultadoLlamada(callSid) {
  try {
    const resultado = await storage.get(`resultado:${callSid}`);
    if (resultado) {
      console.log(`?? Resultado consultado: ${resultado.telefono} - ${resultado.resultado}`);
      estadoMejorado.resultados.set(callSid, resultado);
      return resultado;
    }
    return null;
  } catch (error) {
    console.error('Error consultando resultado:', error);
    return null;
  }
}

// ========================= FUNCIONES ORIGINALES (MANTIENE CDIGO EXISTENTE) =========================
function validateEnv() {
  const req = [
    'JAMBONZ_ACCOUNT_SID',
    'JAMBONZ_API_KEY',
    'JAMBONZ_BASE_URL',
    'APPLICATION_SID',
    'WS_BASE_URL',
    'HTTP_BASE_URL'
  ];
  const missing = req.filter(k => !process.env[k]);
  if (missing.length) {
    console.error('? Faltan variables de entorno:', missing.join(', '));
  }
}
validateEnv();

const sleep = (ms) => new Promise(r => setTimeout(r, ms));

const normalizePhone = (raw) => {
  if (!raw) return '';
  let phone = String(raw).replace(/\+|\s|-/g, '');
  if (phone.startsWith('57') && phone.length > 10) phone = phone.substring(2);
  return phone;
};

// ========================= HISTORIAL ORIGINAL (MANTIENE CDIGO EXISTENTE) =========================
async function cargarNumerosProcesados() {
  try {
    if (fs.existsSync(PROCESSED_FILE)) {
      const data = await fs.promises.readFile(PROCESSED_FILE, 'utf8');
      const parsed = JSON.parse(data);
      numerosProcesados = {
        ultimaEjecucion: parsed.ultimaEjecucion || null,
        numerosLlamados: new Set(parsed.numerosLlamados || []),
        intentos: new Map(Object.entries(parsed.intentos || {}))
      };
      
      // NUEVO: Cargar estado mejorado si existe
      if (parsed.estadoMejorado) {
        estadoMejorado.exitosos = new Set(parsed.estadoMejorado.exitosos || []);
        estadoMejorado.errores = new Map(Object.entries(parsed.estadoMejorado.errores || {}).map(([k, v]) => [k, {
          status: v.status,
          count: v.count,
          lastAttempt: v.lastAttempt ? new Date(v.lastAttempt) : null
        }] ));
      }
    }
  } catch {
    console.log('??  No se pudo cargar historial de llamadas, comenzando desde cero');
  }
}

async function guardarNumerosProcesados() {
  try {
    const data = {
      ultimaEjecucion: new Date().toISOString(),
      numerosLlamados: Array.from(numerosProcesados.numerosLlamados),
      intentos: Object.fromEntries(numerosProcesados.intentos),
      // NUEVO: Guardar estado mejorado
      estadoMejorado: {
        exitosos: Array.from(estadoMejorado.exitosos),
        errores: Object.fromEntries(Array.from(estadoMejorado.errores.entries()).map(([k, v]) => [k, {
          status: v.status,
          count: v.count,
          lastAttempt: v.lastAttempt ? v.lastAttempt.toISOString() : null
        }]))
      }
    };
    await fs.promises.writeFile(PROCESSED_FILE, JSON.stringify(data, null, 2));
  } catch (error) {
    console.error('??  Error guardando historial de llamadas:', error);
  }
}

function yaFueLlamado(telefono) {
  return numerosProcesados.numerosLlamados.has(telefono);
}

function marcarComoLlamado(telefono) {
  numerosProcesados.numerosLlamados.add(telefono);
  const intentos = (numerosProcesados.intentos.get(telefono) || 0) + 1;
  numerosProcesados.intentos.set(telefono, intentos);
}

async function reiniciarHistorial() {
  numerosProcesados = {
    ultimaEjecucion: null,
    numerosLlamados: new Set(),
    intentos: new Map()
  };
  estadoMejorado = {
    errores: new Map(),
    exitosos: new Set(),
    resultados: new Map()
  };
  try {
    if (fs.existsSync(PROCESSED_FILE)) {
      await fs.promises.unlink(PROCESSED_FILE);
    }
    console.log('?? Historial de llamadas reiniciado');
  } catch (error) {
    console.error('??  Error reiniciando historial:', error);
  }
}

// ========================= CALL INFO ORIGINAL (MANTIENE CDIGO EXISTENTE) =========================
async function almacenarDatosContacto(contacto) {
  try {
    const telefono = normalizePhone(contacto.telefono);
    await storage.set(`callinfo:${telefono}`, {
      identificacion:   contacto.identificacion   || '',
      nombrecompleto:   contacto.nombrecompleto   || '',
      referencia:       contacto.referencia       || '',
      monto_inicial:    contacto.monto_inicial    || '',
      cuenta:           contacto.cuenta           || '',
      fecha_vencimiento:contacto.fecha_vencimiento|| '',
      dias_mora:        contacto.dias_mora        || '',
      origen:           contacto.origen           || '',
      tipo_obligacion:  contacto.tipo_obligacion  || '',
      plan:             contacto.plan             || '',
      // NUEVO: Campos adicionales para el prompt mejorado
      plazo_acuerdo:    contacto.plazo_acuerdo    || 'prximos 5 das hbiles',
      min:              contacto.min              || '',
      descuento:        contacto.descuento        || ''
    });
    return true;
  } catch (error) {
    console.error('??  Error almacenando datos del contacto:', error);
    return false;
  }
}

// ========================= WAITERS MEJORADOS (EXTIENDE FUNCIONALIDAD) =========================
function waitCallCompletion(callSid, telefono, timeoutMs = 15 * 60 * 1000) {
  return new Promise(async (resolve) => {
    if (COMPLETED.has(callSid)) {
      COMPLETED.delete(callSid);
      if (llamadasActivas > 0) llamadasActivas--;
      const resultado = await consultarResultadoLlamada(callSid);
      return resolve({ callSid, telefono, reason: 'completed-early', resultado });
    }

    const timer = setTimeout(async () => {
      PENDING.delete(callSid);
      if (llamadasActivas > 0) llamadasActivas--;
      const resultado = await consultarResultadoLlamada(callSid);
      resolve({ callSid, telefono, reason: 'timeout', resultado });
    }, timeoutMs);

    PENDING.set(callSid, async () => {
      clearTimeout(timer);
      if (llamadasActivas > 0) llamadasActivas--;
      const resultado = await consultarResultadoLlamada(callSid);
      resolve({ callSid, telefono, reason: 'completed', resultado });
    });
  });
}

// ========================= LAUNCH CALL MEJORADO (EXTIENDE FUNCIONALIDAD) =========================
async function launchCall(contact) {
  const numero = normalizePhone(contact.telefono);

  if (yaFueExitoso(numero)) {
    console.log(`? Saltando ${numero} - Ya fue exitoso`);
    return null;
  }

  if (!puedeReintentar(numero)) {
    const errorInfo = estadoMejorado.errores.get(numero);
    console.log(`?? Saltando ${numero} - Mximo de reintentos alcanzado (${errorInfo.count}/${MAX_REINTENTOS})`);
    return null;
  }

  if (yaFueLlamado(numero)) {
    const intentos = numerosProcesados.intentos.get(numero) || 0;
    console.log(`??  Saltando ${numero} - Ya fue llamado ${intentos} vez(es)`);
    return null;
  }
  
  // cargar config para este contacto
  await cargarConfiguracionDesdeRedis(contact.company_id);

  await almacenarDatosContacto(contact);
  
  let client;
  try {
    client = getJambonzClient();
  } catch (err) {
    console.error(`? No se pudo obtener cliente Jambonz para ${numero}:`, err.message);
    marcarError(numero, 500);
    await guardarNumerosProcesados();
    return null;
  }

  // leer campos de masterConfig con helper (soporta .value y plano)
  const applicationSid = getCfg('APPLICATION_SID') || process.env.APPLICATION_SID;
  const fromOverride = getCfg('CALLER_ID_OVERRIDE') || process.env.CALLER_ID_OVERRIDE ;
  const sipDomain = getCfg('SIP_TRUNK_DOMAIN')|| process.env.SIP_TRUNK_DOMAIN;
  const wsBase = getCfg('WS_BASE_URL') || process.env.WS_BASE_URL;
  const httpBase = getCfg('HTTP_BASE_URL') || process.env.HTTP_BASE_URL;
  const language = getCfg('CALLING_PARTY_LANGUAGE') || process.env.CALLING_PARTY_LANGUAGE || 'es-PE';

  // Validaciones crticas
  if (!fromOverride) {
    console.error(`? Saltando ${numero}: falta CALLER_ID_OVERRIDE (from).`);
    marcarError(numero, 400);
    await guardarNumerosProcesados();
    return null;
  }
  if (!applicationSid) {
    console.error(`? Saltando ${numero}: falta APPLICATION_SID en la config.`);
    marcarError(numero, 400);
    await guardarNumerosProcesados();
    return null;
  }
  if (!sipDomain) {
    console.warn(`? Advertencia: SIP_TRUNK_DOMAIN no definido; payload SIP puede no ser vlido.`);
  }

  console.log('?? MasterConfig usada para llamada:', {
    applicationSid: !!applicationSid,
    fromOverride,
    sipDomain,
    wsBase: !!wsBase,
    httpBase: !!httpBase
  });

  const payload = {
    application_sid: applicationSid,
    from: "bot4tm",
    to: { type: 'sip', sipUri: `sip:7757${numero}@${sipDomain}` },
    call_hook:       { url: `${wsBase}/call`,    method: 'GET'  },
    call_status_hook:{ url: `${httpBase}/status`, method: 'POST' },
    speech_synthesis_vendor:  'google',
    speech_synthesis_language: language,
    speech_synthesis_voice:   'es-US-Standard-A',
    speech_recognizer_vendor: 'google',
    speech_recognizer_language: language
  };

  try {
    console.log(`?? Llamando a: ${numero} | Contacto: ${contact.nombrecompleto || '-'} | Deuda: ${contact.monto_inicial || '-'}`);
    console.log('?? Payload sintetizado:', { from: payload.from, to: payload.to.sipUri, application_sid: payload.application_sid });
    const response = await client.calls.create(payload);
    const callSid = response && (response.sid || response.call_sid || response.id);

    marcarComoLlamado(numero);
    await guardarNumerosProcesados();

    llamadasActivas++;
    console.log(`?? Llamada iniciada - call_sid: ${callSid}`);

    const done = waitCallCompletion(callSid, numero);
    return { callSid, done, numero };

  } catch (err) {
    const errorMsg = err?.message || String(err);
    const status = err?.response?.status || 500;
    console.error(`? Error al llamar a ${numero}:`, errorMsg);
    if (err?.response?.data) console.error('     ? Response data:', JSON.stringify(err.response.data));
    
    if (status === 503) {
      marcarError(numero, 503);
    } else {
      marcarError(numero, status);
    }

    await guardarNumerosProcesados();
    return null;
  }
}

// ========================= NUEVA VERSIN MEJORADA =========================
function notifyCallCompleted(callSid, telefono, status = 200, transfer = false) {
  const done = PENDING.get(callSid);
  
  // Determinar tipo de resultado
  let resultadoTipo = 'desconocido';
  if (transfer) {
    resultadoTipo = 'transferido';
  } else if (status === 200) {
    resultadoTipo = 'exitoso';
  } else {
    resultadoTipo = 'fallido';
  }

  console.log(`[NOTIFY] Llamada ${callSid} completada: ${telefono} - ${resultadoTipo} (status: ${status})`);
  
  if (status === 200 && telefono) {
    if (transfer) {
      console.log(` ${telefono}: TRANSFERIDO a agente - no se reintentar`);
      marcarExitoso(telefono);
    } else {
      console.log(` ${telefono}: COMPLETADA exitosamente - no se reintentar`);
      marcarExitoso(telefono);
    }
    
    // Guardar resultado detallado
    const resultado = {
      telefono,
      resultado: resultadoTipo,
      detalles: transfer ? 'Llamada transferida a agente humano' : 'Llamada completada exitosamente',
      timestamp: new Date().toISOString(),
      call_sid: callSid,
      status: status
    };
    
    estadoMejorado.resultados.set(callSid, resultado);
    storage.set(`resultado:${callSid}`, resultado).catch(console.error);
  } else if (telefono) {
    console.log(`? ${telefono}: FALLIDA - Se reintentar en siguiente ronda`);
    marcarError(telefono, status);
  }
  
  // Resolver la promesa de espera
  if (done) {
    PENDING.delete(callSid);
    done();
  } else {
    COMPLETED.add(callSid);
  }
  
  // Guardar cambios en el estado
  guardarNumerosProcesados().catch(console.error);
}

// ========================= NUEVA FUNCIN: CLASIFICAR RESULTADOS =========================
function clasificarResultadoLlamada(telefono, callStatus, duration, motivo) {
  
  // TRANSFERENCIAS - XITO, NO REINTENTAR
  if (motivo && (motivo.includes('transfer') || motivo === 'agent_transfer')) {
    return { 
      tipo: 'transferido', 
      exitoso: true, 
      reintentar: false,
      descripcion: 'Llamada transferida a agente humano'
    };
  }
  
  // LLAMADAS COMPLETADAS
  if (callStatus === 'completed') {
    if (duration > 60) {
      return { 
        tipo: 'exitoso_largo', 
        exitoso: true, 
        reintentar: false,
        descripcion: 'Llamada larga completada exitosamente'
      };
    } else if (duration > 30) {
      return { 
        tipo: 'exitoso', 
        exitoso: true, 
        reintentar: false,
        descripcion: 'Llamada completada exitosamente'
      };
    } else if (duration > 15) {
      return { 
        tipo: 'cortada', 
        exitoso: false, 
        reintentar: true,
        descripcion: 'Llamada cortada prematuramente'
      };
    } else {
      return { 
        tipo: 'muy_corta', 
        exitoso: false, 
        reintentar: true,
        descripcion: 'Llamada muy corta, posible buzn'
      };
    }
  }
  
  // NO CONTESTA
  if (callStatus === 'no-answer') {
    return { 
      tipo: 'no_contesta', 
      exitoso: false, 
      reintentar: true,
      descripcion: 'No contest la llamada'
    };
  }
  
  // OCUPADO
  if (callStatus === 'busy') {
    return { 
      tipo: 'ocupado', 
      exitoso: false, 
      reintentar: true,
      descripcion: 'Nmero ocupado'
    };
  }
  
  // FALLIDAS
  if (callStatus === 'failed') {
    return { 
      tipo: 'fallida', 
      exitoso: false, 
      reintentar: true,
      descripcion: 'Llamada fallida por error tcnico'
    };
  }
  
  // CANCELADA
  if (callStatus === 'canceled') {
    return { 
      tipo: 'cancelada', 
      exitoso: false, 
      reintentar: true,
      descripcion: 'Llamada cancelada'
    };
  }
  
  // DESCONOCIDO
  return { 
    tipo: 'desconocido', 
    exitoso: false, 
    reintentar: true,
    descripcion: 'Estado de llamada no reconocido'
  };
}


// ========================= PROCESAR LOTE ORIGINAL (MANTIENE CDIGO EXISTENTE) =========================
async function procesarLote(lote, numeroDeLote) {
  console.log(`\n?? PROCESANDO LOTE ${numeroDeLote} (${lote.length} contactos)`);
  console.log('-'.repeat(50));

  const waiters = [];
  let realizadas = 0;

  for (let i = 0; i < lote.length; i++) {
    if (detenerLlamadas) {
      console.log('??  Ejecucin detenida por usuario');
      break;
    }

    const contacto = lote[i];
    console.log(`?? Procesando contacto ${i + 1}/${lote.length}: ${contacto.telefono}`);

    const res = await launchCall(contacto);
    
    if (res && res.done) {
      realizadas++;
      waiters.push(res.done);
    console.log(`?? Llamada iniciada para ${contacto.telefono} - Esperando finalizacin`);
    } else {
      console.log(`?? No se pudo iniciar llamada para ${contacto.telefono}`);
    }

    // pequeo respiro para evitar rate-limits (ajusta si quieres)
    if (i < lote.length - 1){
    console.log('?? Esperando 1 segundo antes de siguiente llamada...');
     await sleep(1000);
    }
  }

  if (waiters.length > 0) {
    console.log(`? Esperando a que terminen ${waiters.length} llamada(s) del lote ${numeroDeLote}...`);
    
    // MEJORADO: Procesar resultados de las llamadas
    const resultados = await Promise.allSettled(waiters.map(w => w));
    
    // NUEVO: Clasificar resultados
    for (const resultado of resultados) {
      if (resultado.status === 'fulfilled') {
        const { telefono, resultado: res } = resultado.value;
        
        if (res && (res.resultado === 'exitoso' || res.resultado === 'buzon' || res.resultado === 'equivocado')) {
         marcarExitoso(telefono);
          console.log(`? ${telefono}: ${res.resultado} - Marcado como exitoso`);
        } else if (res && res.resultado === 'fallido') {
          console.log(`?? ${telefono}: Fallida - Se reintentar en siguiente ronda`);
        } else {
          console.log(`?? ${telefono}: Estado desconocido - ${JSON.stringify(res)}`);
        }
      } else {
        console.log(`?? Llamada fall:`, resultado.reason);
      }
    }
  } else {
    console.log(`??  Lote ${numeroDeLote} sin llamadas iniciadas`);
  }

  lotesCompletados++;
  
  console.log(`? LOTE ${numeroDeLote} COMPLETADO | Llamadas realizadas: ${realizadas}/${lote.length}`);
  console.log('-'.repeat(50));
}



// ========================= NUEVO: EJECUCIN POR RONDAS INTELIGENTES =========================
async function ejecutarRondasInteligentes(contactos) {
  let ronda = 1;
  let contactosPendientes = contactos.filter(c => {
    const numero = normalizePhone(c.telefono);
    return !yaFueExitoso(numero) && puedeReintentar(numero);
  });

  console.log(`?? INICIANDO SISTEMA DE RONDAS - Contactos pendientes: ${contactosPendientes.length}`);
  console.log(`?? Configuracin: Lotes de ${BATCH_SIZE}, Mximo ${MAX_REINTENTOS} reintentos`);
  
 // while (contactosPendientes.length > 0 && !detenerLlamadas) {
   while (contactosPendientes.length > 0 && !detenerLlamadas && ronda <= MAX_REINTENTOS + 1) {
    console.log(`\n?? RONDA ${ronda} - Contactos pendientes: ${contactosPendientes.length}`);
    
    // Dividir en lotes (mantiene lgica original)
    const lotes = [];
    for (let i = 0; i < contactosPendientes.length; i += BATCH_SIZE) {
      lotes.push(contactosPendientes.slice(i, i + BATCH_SIZE));
    }
    
    console.log(`?? Procesando ${lotes.length} lote(s) en ronda ${ronda}`);

    // Procesar lotes (mantiene lgica original)
    for (let i = 0; i < lotes.length; i++) {
      if (detenerLlamadas) {
        console.log('??  Ejecucin detenida por usuario');
        break;
      }
      console.log(`\n?? Procesando lote ${i + 1}/${lotes.length} (${lotes[i].length} contactos)`);
      //await procesarLote(lotes[i], i + 1);
      await procesarLote(lotes[i], `R${ronda}-L${i + 1}`);
      await guardarNumerosProcesados();
      
      //if (i < lotes.length - 1) {
    if (i < lotes.length - 1 && !detenerLlamadas) {
        console.log('??  Esperando 5 segundos antes del prximo lote...\n');
        await sleep(5000);
      }
    }

    // Actualizar lista para siguiente ronda
    const pendientesAntes = contactosPendientes.length;
    contactosPendientes = contactos.filter(c => {
      const numero = normalizePhone(c.telefono);
      return !yaFueExitoso(numero) && puedeReintentar(numero);
    });
    
    console.log(`?? Ronda ${ronda} completada. Pendientes: ${pendientesAntes} -> ${contactosPendientes.length}`);
    
   // if (contactosPendientes.length > 0) {
   if (contactosPendientes.length > 0 && !detenerLlamadas) {
      console.log(`? Esperando 15 segundos antes de la prxima ronda...`);
      await sleep(15000);
    }
    
    ronda++;
    
    if (ronda > MAX_REINTENTOS + 2) {
      console.log('?? Mximo de rondas alcanzado, terminando ejecucin');
      break;
    }
  }
  console.log(`?? Sistema de rondas finalizado despus de ${ronda - 1} rondas`);
  return ronda - 1; // Retorna nmero de rondas completadas
}

// ========================= EJECUTAR LLAMADAS MEJORADO (EXTIENDE FUNCIONALIDAD) =========================
/*async function ejecutarLlamadas() {
  await cargarNumerosProcesados();

  return new Promise((resolve, reject) => {
  
  try {
  
    const contactos = await obtenerContactos();;
    const lotes = [];

    console.log('?? Leyendo archivo de contactos...');
    console.log(`??  Nmeros ya procesados: ${numerosProcesados.numerosLlamados.size}`);
    console.log(`?  Nmeros exitosos: ${estadoMejorado.exitosos.size}`);
    console.log(`??  Nmeros con error: ${estadoMejorado.errores.size}`);

    if (contactos.length === 0) {
        console.log('?? No hay contactos para procesar');
        return resolve();
       }
       
       
    fs.createReadStream(CONTACTS_FILE)
      .pipe(csv())
      .on('data', (row) => {
        if (row.telefono && String(row.telefono).trim() !== '') {
          contactos.push(row);
        }
      })
      .on('end', async () => {
        console.log(`?? Total de contactos en CSV: ${contactos.length}`);

        // NUEVO: Ejecutar rondas inteligentes
        const rondasCompletadas = await ejecutarRondasInteligentes(contactos);

        // MANTIENE RESUMEN ORIGINAL (extendido)
        console.log('\n? EJECUCIN COMPLETADA');
        console.log(`?? Resumen:`);
        console.log(`    Contactos en CSV:            ${contactos.length}`);
        console.log(`    Nmeros procesados totales:  ${numerosProcesados.numerosLlamados.size}`);
        console.log(`    Nmeros exitosos:            ${estadoMejorado.exitosos.size}`);
        console.log(`    Nmeros con error:           ${estadoMejorado.errores.size}`);
        console.log(`    Rondas completadas:          ${rondasCompletadas}`);
        console.log(`    Lotes procesados:            ${lotesCompletados}`);

        // NUEVO: Mostrar errores permanentes
        const erroresPermanentes = Array.from(estadoMejorado.errores.entries())
          .filter(([_, info]) => info.count >= MAX_REINTENTOS);
        
        if (erroresPermanentes.length > 0) {
          console.log(`    Errores permanentes:         ${erroresPermanentes.length}`);
          console.log('    Nmeros con error permanente:');
          erroresPermanentes.forEach(([numero, info]) => {
            console.log(`     - ${numero}: Status ${info.status}, Intentos: ${info.count}`);
          });
        }

        resolve();
        }catch(error){
          reject(error)
        }
      })
      .on('error', reject);
  });
}*/

async function ejecutarLlamadas() {
  await cargarNumerosProcesados();

  return new Promise(async (resolve, reject) => {
    try {
      console.log('?? Leyendo archivo de contactos...');
      console.log(`??  Nmeros ya procesados: ${numerosProcesados.numerosLlamados.size}`);
      console.log(`?  Nmeros exitosos: ${estadoMejorado.exitosos.size}`);
      console.log(`??  Nmeros con error: ${estadoMejorado.errores.size}`);

      // Obtener contactos directamente - MS SIMPLE
      const contactos = await obtenerContactos();
      console.log(`?? Total de contactos obtenidos: ${contactos.length}`);

      if (contactos.length === 0) {
        console.log('?? No hay contactos para procesar');
        return resolve();
      }

      // NUEVO: Ejecutar rondas inteligentes
      const rondasCompletadas = await ejecutarRondasInteligentes(contactos);

      // MANTIENE RESUMEN ORIGINAL (extendido)
      console.log('\n? EJECUCIN COMPLETADA');
      console.log(`?? Resumen:`);
      console.log(`    Contactos en CSV:            ${contactos.length}`);
      console.log(`    Nmeros procesados totales:  ${numerosProcesados.numerosLlamados.size}`);
      console.log(`    Nmeros exitosos:            ${estadoMejorado.exitosos.size}`);
      console.log(`    Nmeros con error:           ${estadoMejorado.errores.size}`);
      console.log(`    Rondas completadas:          ${rondasCompletadas}`);
      console.log(`    Lotes procesados:            ${lotesCompletados}`);

      // NUEVO: Mostrar errores permanentes
      const erroresPermanentes = Array.from(estadoMejorado.errores.entries())
        .filter(([_, info]) => info.count >= MAX_REINTENTOS);
      
      if (erroresPermanentes.length > 0) {
        console.log(`    Errores permanentes:         ${erroresPermanentes.length}`);
        console.log('    Nmeros con error permanente:');
        erroresPermanentes.forEach(([numero, info]) => {
          console.log(`     - ${numero}: Status ${info.status}, Intentos: ${info.count}`);
        });
      }

      resolve();
    } catch (error) {
      console.error('?? Error en ejecutarLlamadas:', error);
      reject(error);
    }
  });
}
// ========================= ESTADO Y CONTROL ORIGINAL (MANTIENE CDIGO EXISTENTE) =========================
function obtenerEstadoLlamadas() {
  return {
    detenerLlamadas,
    llamadasActivas,
    lotesCompletados,
    numerosProcesados: numerosProcesados.numerosLlamados.size,
    // NUEVO: Campos extendidos
    numerosExitosos: estadoMejorado.exitosos.size,
    numerosConError: estadoMejorado.errores.size,
    estado: detenerLlamadas ? 'detenido' : (llamadasActivas > 0 ? 'en_ejecucion' : 'completado')
  };
}

function detenerEjecucionLlamadas() {
  detenerLlamadas = true;
  console.log('??  Se marc bandera para detener ejecucin.');
}

async function reiniciarEjecucion() {
  detenerLlamadas = false;
  llamadasActivas = 0;
  lotesCompletados = 0;
  console.log('?? Ejecucin reiniciada.');
}

function verEstadisticas() {
  console.log('\n?? ESTADSTICAS DE LLAMADAS');
  console.log('-'.repeat(40));
  console.log(` Nmeros procesados: ${numerosProcesados.numerosLlamados.size}`);
  console.log(` Nmeros exitosos: ${estadoMejorado.exitosos.size}`);
  console.log(` Nmeros con error: ${estadoMejorado.errores.size}`);
  console.log(` ltima ejecucin: ${numerosProcesados.ultimaEjecucion || 'Nunca'}`);
  
  console.log(' Intentos por nmero (top 5):');
  const sortedIntentos = Array.from(numerosProcesados.intentos.entries())
    .sort((a, b) => b[1] - a[1])
    .slice(0, 5);
  sortedIntentos.forEach(([numero, intentos]) => {
    console.log(`  - ${numero}: ${intentos} intento(s)`);
  });
}

// ========================= EXPORTS ORIGINAL (MANTIENE CDIGO EXISTENTE) =========================
module.exports = {
  ejecutarLlamadas,
  detenerEjecucionLlamadas,
  obtenerEstadoLlamadas,
  reiniciarEjecucion,
  reiniciarHistorial,
  verEstadisticas,
  cargarNumerosProcesados,
  notifyCallCompleted, 
  clasificarResultadoLlamada
};


