require('dotenv').config();
const fs = require('fs');
const path = require('path');
const csv = require('csv-parser');
const jambonz = require('@jambonz/node-client');
const client = jambonz(process.env.JAMBONZ_ACCOUNT_SID, process.env.JAMBONZ_API_KEY, {
  baseUrl: process.env.JAMBONZ_BASE_URL
});

const CONTACTS_FILE = path.join(__dirname, '../data/contactos.csv');
const BATCH_SIZE = 5; // Lotes de 5 nmeros

let detenerLlamadas = false;
let llamadasActivas = 0;
let lotesCompletados = 0;

const normalizePhone = (raw) => {
  let phone = raw.replace(/\+|\s|-/g, '');
  if (phone.startsWith('57') && phone.length > 10) phone = phone.substring(2);
  return phone;
};

const launchCall = async (contact) => {
  const numero = normalizePhone(contact.telefono);

  const payload = {
    application_sid: process.env.APPLICATION_SID,
    from: 'bot4tm',
    to: {
      type: 'sip',
      sipUri: `sip:7757${numero}@atmbponextcall.controlnextapp.com`
    },
    call_hook: {
      url: `${process.env.WS_BASE_URL}/call`,
      method: 'GET'
    },
    call_status_hook: {
      url: `${process.env.HTTP_BASE_URL}/status`,
      method: 'POST'
    },
    speech_synthesis_vendor: 'google',
    speech_synthesis_language: 'es-US',
    speech_synthesis_voice: 'es-US-Standard-A',
    speech_recognizer_vendor: 'google',
    speech_recognizer_language: 'es-CO'
  };

  try {
    llamadasActivas++;
    console.log(`[LOTE ${lotesCompletados + 1}] Iniciando llamada ${llamadasActivas}/5 a: ${numero}`);
    console.log(`   Contacto: ${contact.nombrecompleto} - Deuda: ${contact.monto_inicial}`);
    
    const response = await client.calls.create(payload);
    console.log(`   ? Llamada iniciada - call_sid: ${response.sid}`);
    
    return response.sid;
  } catch (err) {
    console.error(`   ? Error al llamar a ${numero}:`, err.message || err);
    llamadasActivas--;
    return null;
  }
};

const procesarLote = async (lote) => {
  console.log(`\n?? PROCESANDO LOTE ${lotesCompletados + 1} (${lote.length} contactos)`);
  console.log('-'.repeat(50));
  
  const resultados = [];
  
  for (const contacto of lote) {
    if (detenerLlamadas) {
      console.log('??  Ejecucin detenida por usuario');
      return;
    }
    
    const callSid = await launchCall(contacto);
    resultados.push({ contacto, callSid, success: !!callSid });
    
    // Esperar entre llamadas (2 segundos)
    if (contacto !== lote[lote.length - 1]) {
      await new Promise(res => setTimeout(res, 2000));
    }
  }
  
  // Esperar a que se completen las 5 llamadas del lote
  console.log(`\n? Esperando finalizacin del lote ${lotesCompletados + 1}...`);
  
  // Simulamos espera (en produccin deberas monitorear el estado real de las llamadas)
  await new Promise(res => setTimeout(res, 30000)); // 30 segundos por lote
  
  lotesCompletados++;
  llamadasActivas = 0;
  
  console.log(`? LOTE ${lotesCompletados} COMPLETADO`);
  console.log('-'.repeat(50));
  
  return resultados;
};

const ejecutarLlamadas = async () => {
  return new Promise((resolve, reject) => {
    const contactos = [];
    let lotes = [];
    
    console.log('?? Leyendo archivo de contactos...');
    
    fs.createReadStream(CONTACTS_FILE)
      .pipe(csv())
      .on('data', (row) => {
        if (row.telefono && row.telefono.trim() !== '') {
          contactos.push(row);
        }
      })
      .on('end', async () => {
        console.log(`?? Total de contactos vlidos: ${contactos.length}`);
        
        // Crear lotes de 5 contactos
        for (let i = 0; i < contactos.length; i += BATCH_SIZE) {
          lotes.push(contactos.slice(i, i + BATCH_SIZE));
        }
        
        console.log(`?? Total de lotes a procesar: ${lotes.length}`);
        
        // Procesar lotes secuencialmente
        for (let i = 0; i < lotes.length; i++) {
          if (detenerLlamadas) {
            console.log('??  Ejecucin detenida por usuario');
            break;
          }
          
          await procesarLote(lotes[i]);
          
          // Esperar entre lotes (5 segundos)
          if (i < lotes.length - 1) {
            console.log('\n? Esperando 5 segundos antes del prximo lote...\n');
            await new Promise(res => setTimeout(res, 5000));
          }
        }
        
        console.log('\n?? TODOS LOS LOTES COMPLETADOS');
        console.log(`?? Resumen:`);
        console.log(`    Contactos procesados: ${contactos.length}`);
        console.log(`    Lotes completados: ${lotesCompletados}`);
        console.log(`    Lotes pendientes: ${lotes.length - lotesCompletados}`);
        
        resolve();
      })
      .on('error', reject);
  });
};

const obtenerEstadoLlamadas = () => {
  return {
    detenerLlamadas,
    llamadasActivas,
    lotesCompletados,
    estado: detenerLlamadas ? 'detenido' : (llamadasActivas > 0 ? 'en_ejecucion' : 'completado')
  };
};

const detenerEjecucionLlamadas = () => {
  detenerLlamadas = true;
  console.log('??  Se marc bandera para detener ejecucin.');
};

const reiniciarEjecucion = () => {
  detenerLlamadas = false;
  llamadasActivas = 0;
  lotesCompletados = 0;
  console.log('?? Ejecucin reiniciada.');
};

module.exports = {
  ejecutarLlamadas,
  detenerEjecucionLlamadas,
  obtenerEstadoLlamadas,
  reiniciarEjecucion
};