require('dotenv').config();
const fs = require('fs');
const path = require('path');
const csv = require('csv-parser');
const jambonz = require('@jambonz/node-client');
const client = jambonz(process.env.JAMBONZ_ACCOUNT_SID, process.env.JAMBONZ_API_KEY, {
  baseUrl: process.env.JAMBONZ_BASE_URL
});

const storage = require('./storage');

const CONTACTS_FILE = path.join(__dirname, '../data/contactos.csv');
const PROCESSED_FILE = path.join(__dirname, '../data/contactos_procesados.json');
const BATCH_SIZE = 5;

let detenerLlamadas = false;
let llamadasActivas = 0;
let lotesCompletados = 0;

// Estructura para guardar nmeros procesados
let numerosProcesados = {
  ultimaEjecucion: null,
  numerosLlamados: new Set(),
  intentos: new Map() // { numero: count }
};

// Cargar nmeros ya procesados
const cargarNumerosProcesados = async () => {
  try {
    if (fs.existsSync(PROCESSED_FILE)) {
      const data = await fs.promises.readFile(PROCESSED_FILE, 'utf8');
      const parsed = JSON.parse(data);
      numerosProcesados = {
        ultimaEjecucion: parsed.ultimaEjecucion,
        numerosLlamados: new Set(parsed.numerosLlamados || []),
        intentos: new Map(Object.entries(parsed.intentos || {}))
      };
    }
  } catch (error) {
    console.log('??  No se pudo cargar historial de llamadas, comenzando desde cero');
  }
};


// Guardar nmeros procesados
const guardarNumerosProcesados = async () => {
  try {
    const data = {
      ultimaEjecucion: new Date().toISOString(),
      numerosLlamados: Array.from(numerosProcesados.numerosLlamados),
      intentos: Object.fromEntries(numerosProcesados.intentos)
    };
    await fs.promises.writeFile(PROCESSED_FILE, JSON.stringify(data, null, 2));
  } catch (error) {
    console.error('? Error guardando historial de llamadas:', error);
  }
};

// Verificar si un nmero ya fue procesado
const yaFueLlamado = (telefono) => {
  return numerosProcesados.numerosLlamados.has(telefono);
};



// Marcar nmero como procesado
const marcarComoLlamado = (telefono) => {
  numerosProcesados.numerosLlamados.add(telefono);
  const intentos = (numerosProcesados.intentos.get(telefono) || 0) + 1;
  numerosProcesados.intentos.set(telefono, intentos);
};

// Reiniciar historial (opcional)
const reiniciarHistorial = async () => {
  numerosProcesados = {
    ultimaEjecucion: null,
    numerosLlamados: new Set(),
    intentos: new Map()
  };
  try {
    if (fs.existsSync(PROCESSED_FILE)) {
      await fs.promises.unlink(PROCESSED_FILE);
    }
    console.log('? Historial de llamadas reiniciado');
  } catch (error) {
    console.error('? Error reiniciando historial:', error);
  }
};

const normalizePhone = (raw) => {
  let phone = raw.replace(/\+|\s|-/g, '');
  if (phone.startsWith('57') && phone.length > 10) phone = phone.substring(2);
  return phone;
};


const almacenarDatosContacto = async (contacto) => {
  try {
    const telefono = normalizePhone(contacto.telefono);
    await storage.set(`callinfo:${telefono}`, {
      identificacion: contacto.identificacion || '',
      nombrecompleto: contacto.nombrecompleto || '',
      monto_inicial: contacto.monto_inicial || '',
      cuenta: contacto.cuenta || '',
      fecha_vencimiento: contacto.fecha_vencimiento || '',
      dias_mora: contacto.dias_mora || '',
      origen: contacto.origen || '',
      tipo_obligacion: contacto.tipo_obligacion || '',
      plan: contacto.plan || ''
    });
    return true;
  } catch (error) {
    console.error('Error almacenando datos del contacto:', error);
    return false;
  }
};

const launchCall = async (contact) => {
  const numero = normalizePhone(contact.telefono);

  // Verificar si ya fue llamado
  if (yaFueLlamado(numero)) {
    const intentos = numerosProcesados.intentos.get(numero) || 0;
    console.log(`??  Saltando nmero ${numero} - Ya fue llamado ${intentos} vez(es) antes`);
    return null;
  }
  
   await almacenarDatosContacto(contact);

  const payload = {
    application_sid: process.env.APPLICATION_SID,
    from: 'bot4tm',
    to: {
      type: 'sip',
      sipUri: `sip:7757${numero}@atmbponextcall.controlnextapp.com`
    },
    call_hook: {
      url: `${process.env.WS_BASE_URL}/call`,
      method: 'GET'
    },
    call_status_hook: {
      url: `${process.env.HTTP_BASE_URL}/status`,
      method: 'POST'
    },
    speech_synthesis_vendor: 'google',
    speech_synthesis_language: 'es-US',
    speech_synthesis_voice: 'es-US-Standard-A',
    speech_recognizer_vendor: 'google',
    speech_recognizer_language: 'es-CO'
  };

  try {
    llamadasActivas++;
    console.log(`[LOTE ${lotesCompletados + 1}] Iniciando llamada ${llamadasActivas}/5 a: ${numero}`);
    console.log(`   Contacto: ${contact.nombrecompleto} - Deuda: ${contact.monto_inicial}`);
    
    const response = await client.calls.create(payload);
    
    // Marcar como llamado INMEDIATAMENTE despus de iniciar la llamada
    marcarComoLlamado(numero);
    await guardarNumerosProcesados();
    
    console.log(`   ? Llamada iniciada - call_sid: ${response.sid}`);
    
    return response.sid;
  } catch (err) {
    console.error(`   ? Error al llamar a ${numero}:`, err.message || err);
    llamadasActivas--;
    return null;
  }
};

const procesarLote = async (lote) => {
  console.log(`\n?? PROCESANDO LOTE ${lotesCompletados + 1} (${lote.length} contactos)`);
  console.log('-'.repeat(50));
  
  const resultados = [];
  let llamadasRealizadas = 0;
  
  for (const contacto of lote) {
    if (detenerLlamadas) {
      console.log('??  Ejecucin detenida por usuario');
      return;
    }
    
    const callSid = await launchCall(contacto);
    if (callSid) {
      llamadasRealizadas++;
      resultados.push({ contacto, callSid, success: true });
    } else {
      resultados.push({ contacto, callSid: null, success: false });
    }
    
    // Esperar entre llamadas (2 segundos) solo si no es la ltima
    if (contacto !== lote[lote.length - 1]) {
      await new Promise(res => setTimeout(res, 2000));
    }
  }
  
  if (llamadasRealizadas > 0) {
    console.log(`\n? Esperando finalizacin del lote ${lotesCompletados + 1}...`);
    await new Promise(res => setTimeout(res, 30000));
  } else {
    console.log(`\n??  Lote ${lotesCompletados + 1} sin nmeros nuevos para llamar`);
  }
  
  lotesCompletados++;
  llamadasActivas = 0;
  
  console.log(`[LOTE ${lotesCompletados + 1}] Iniciando llamada ${llamadasActivas}/${BATCH_SIZE} a: ${numero}`);
  console.log(`   Contacto: ${contact.nombrecompleto} - Deuda: ${contact.monto_inicial}`);
  
  console.log(`? LOTE ${lotesCompletados} COMPLETADO`);
  console.log(`   ?? Llamadas realizadas: ${llamadasRealizadas}/${lote.length}`);
  console.log('-'.repeat(50));
  

  
  return resultados;
};

const ejecutarLlamadas = async () => {
  // Cargar historial primero
  await cargarNumerosProcesados();
  
  return new Promise((resolve, reject) => {
    const contactos = [];
    let lotes = [];
    
    console.log('?? Leyendo archivo de contactos...');
    console.log(`?? Nmeros ya procesados: ${numerosProcesados.numerosLlamados.size}`);
    
    fs.createReadStream(CONTACTS_FILE)
      .pipe(csv())
      .on('data', (row) => {
        if (row.telefono && row.telefono.trim() !== '') {
          contactos.push(row);
        }
      })
      .on('end', async () => {
        console.log(`?? Total de contactos en CSV: ${contactos.length}`);
        
        // Filtrar contactos que NO han sido llamados
        const contactosNuevos = contactos.filter(contacto => 
          !yaFueLlamado(normalizePhone(contacto.telefono))
        );
        
        console.log(`?? Contactos nuevos por llamar: ${contactosNuevos.length}`);
        
        if (contactosNuevos.length === 0) {
          console.log('?? Todos los nmeros ya han sido procesados anteriormente');
          return resolve();
        }
        
        // Crear lotes de 5 contactos
        for (let i = 0; i < contactosNuevos.length; i += BATCH_SIZE) {
          lotes.push(contactosNuevos.slice(i, i + BATCH_SIZE));
        }
        
        console.log(`?? Total de lotes a procesar: ${lotes.length}`);
        
        // Procesar lotes secuencialmente
        for (let i = 0; i < lotes.length; i++) {
          if (detenerLlamadas) {
            console.log('??  Ejecucin detenida por usuario');
            break;
          }
          
          await procesarLote(lotes[i]);
          
          // Guardar progreso despus de cada lote
          await guardarNumerosProcesados();
          
          // Esperar entre lotes (5 segundos)
          if (i < lotes.length - 1) {
            console.log('\n? Esperando 5 segundos antes del prximo lote...\n');
            await new Promise(res => setTimeout(res, 5000));
          }
        }
        
        console.log('\n?? TODOS LOS LOTES COMPLETADOS');
        console.log(`?? Resumen:`);
        console.log(`    Contactos en CSV: ${contactos.length}`);
        console.log(`    Contactos nuevos: ${contactosNuevos.length}`);
        console.log(`    Lotes procesados: ${lotesCompletados}`);
        console.log(`    Nmeros procesados totales: ${numerosProcesados.numerosLlamados.size}`);
        
        resolve();
      })
      .on('error', reject);
  });
};

const obtenerEstadoLlamadas = () => {
  return {
    detenerLlamadas,
    llamadasActivas,
    lotesCompletados,
    numerosProcesados: numerosProcesados.numerosLlamados.size,
    estado: detenerLlamadas ? 'detenido' : (llamadasActivas > 0 ? 'en_ejecucion' : 'completado')
  };
};

const detenerEjecucionLlamadas = () => {
  detenerLlamadas = true;
  console.log('??  Se marc bandera para detener ejecucin.');
};

const reiniciarEjecucion = async () => {
  detenerLlamadas = false;
  llamadasActivas = 0;
  lotesCompletados = 0;
  console.log('?? Ejecucin reiniciada.');
};

// Nueva funcin para ver estadsticas
const verEstadisticas = () => {
  console.log('\n?? ESTADSTICAS DE LLAMADAS');
  console.log('-'.repeat(30));
  console.log(` Nmeros procesados: ${numerosProcesados.numerosLlamados.size}`);
  console.log(` ltima ejecucin: ${numerosProcesados.ultimaEjecucion || 'Nunca'}`);
  console.log(` Intentos por nmero:`);
  
  // Mostrar top 5 nmeros con ms intentos
  const sortedIntentos = Array.from(numerosProcesados.intentos.entries())
    .sort((a, b) => b[1] - a[1])
    .slice(0, 5);
  
  sortedIntentos.forEach(([numero, intentos]) => {
    console.log(`  - ${numero}: ${intentos} intento(s)`);
  });
};

module.exports = {
  ejecutarLlamadas,
  detenerEjecucionLlamadas,
  obtenerEstadoLlamadas,
  reiniciarEjecucion,
  reiniciarHistorial,
  verEstadisticas,
  cargarNumerosProcesados
};