// caller.js

require('dotenv').config();

const fs = require('fs');

const path = require('path');

const csv = require('csv-parser');

const jambonz = require('@jambonz/node-client');



const client = jambonz(

  process.env.JAMBONZ_ACCOUNT_SID,

  process.env.JAMBONZ_API_KEY,

  { baseUrl: process.env.JAMBONZ_BASE_URL }

);



const storage = require('./storage');



// ========================= Constantes / Estado =========================

const CONTACTS_FILE  = path.join(__dirname, '../data/contactos.csv');

const PROCESSED_FILE = path.join(__dirname, '../data/contactos_procesados.json');

const BATCH_SIZE = 5;



let detenerLlamadas = false;

let llamadasActivas = 0;

let lotesCompletados = 0;



// Historial de nmeros procesados

let numerosProcesados = {

  ultimaEjecucion: null,

  numerosLlamados: new Set(),          // Set<string>

  intentos: new Map()                  // Map<string, number>

};



// Esperas por finalizacin de llamadas

const PENDING   = new Map(); // callSid -> resolver()

const COMPLETED = new Set(); // callSid que termin antes de que armemos el waiter



// ========================= Utils =========================

function validateEnv() {

  const req = [

    'JAMBONZ_ACCOUNT_SID',

    'JAMBONZ_API_KEY',

    'JAMBONZ_BASE_URL',

    'APPLICATION_SID',

    'WS_BASE_URL',

    'HTTP_BASE_URL'

  ];

  const missing = req.filter(k => !process.env[k]);

  if (missing.length) {

    console.error('? Faltan variables de entorno:', missing.join(', '));

  }

}

validateEnv();



const sleep = (ms) => new Promise(r => setTimeout(r, ms));



const normalizePhone = (raw) => {

  if (!raw) return '';

  let phone = String(raw).replace(/\+|\s|-/g, '');

  // Si viene con 57 y tiene ms de 10, recorta el 57 inicial

  if (phone.startsWith('57') && phone.length > 10) phone = phone.substring(2);

  return phone;

};



// ========================= Historial llamadas (disco) =========================

async function cargarNumerosProcesados() {

  try {

    if (fs.existsSync(PROCESSED_FILE)) {

      const data = await fs.promises.readFile(PROCESSED_FILE, 'utf8');

      const parsed = JSON.parse(data);

      numerosProcesados = {

        ultimaEjecucion: parsed.ultimaEjecucion || null,

        numerosLlamados: new Set(parsed.numerosLlamados || []),

        intentos: new Map(Object.entries(parsed.intentos || {}))

      };

    }

  } catch {

    console.log('??  No se pudo cargar historial de llamadas, comenzando desde cero');

  }

}



async function guardarNumerosProcesados() {

  try {

    const data = {

      ultimaEjecucion: new Date().toISOString(),

      numerosLlamados: Array.from(numerosProcesados.numerosLlamados),

      intentos: Object.fromEntries(numerosProcesados.intentos)

    };

    await fs.promises.writeFile(PROCESSED_FILE, JSON.stringify(data, null, 2));

  } catch (error) {

    console.error('??  Error guardando historial de llamadas:', error);

  }

}



function yaFueLlamado(telefono) {

  return numerosProcesados.numerosLlamados.has(telefono);

}



function marcarComoLlamado(telefono) {

  numerosProcesados.numerosLlamados.add(telefono);

  const intentos = (numerosProcesados.intentos.get(telefono) || 0) + 1;

  numerosProcesados.intentos.set(telefono, intentos);

}



async function reiniciarHistorial() {

  numerosProcesados = {

    ultimaEjecucion: null,

    numerosLlamados: new Set(),

    intentos: new Map()

  };

  try {

    if (fs.existsSync(PROCESSED_FILE)) {

      await fs.promises.unlink(PROCESSED_FILE);

    }

    console.log('?? Historial de llamadas reiniciado');

  } catch (error) {

    console.error('??  Error reiniciando historial:', error);

  }

}



// ========================= Call info en storage =========================

async function almacenarDatosContacto(contacto) {

  try {

    const telefono = normalizePhone(contacto.telefono);

    await storage.set(`callinfo:${telefono}`, {

      identificacion:   contacto.identificacion   || '',

      nombrecompleto:   contacto.nombrecompleto   || '',

      referencia    :   contacto.referencia       || '',

      monto_inicial:    contacto.monto_inicial    || '',

      cuenta:           contacto.cuenta           || '',

      fecha_vencimiento:contacto.fecha_vencimiento|| '',

      dias_mora:        contacto.dias_mora        || '',

      origen:           contacto.origen           || '',

      tipo_obligacion:  contacto.tipo_obligacion  || '',

      plan:             contacto.plan             || ''

    });

    return true;

  } catch (error) {

    console.error('??  Error almacenando datos del contacto:', error);

    return false;

  }

}



// ========================= Waiters de finalizacin =========================

function waitCallCompletion(callSid, timeoutMs = 15 * 60 * 1000) {

  return new Promise((resolve) => {

    // Si ya lleg el cierre ANTES de armar el waiter

    if (COMPLETED.has(callSid)) {

      COMPLETED.delete(callSid);

      if (llamadasActivas > 0) llamadasActivas--;

      return resolve({ callSid, reason: 'completed-early' });

    }



    const timer = setTimeout(() => {

      PENDING.delete(callSid);

      if (llamadasActivas > 0) llamadasActivas--;

      resolve({ callSid, reason: 'timeout' });

    }, timeoutMs);



    PENDING.set(callSid, () => {

      clearTimeout(timer);

      if (llamadasActivas > 0) llamadasActivas--;

      resolve({ callSid, reason: 'completed' });

    });

  });

}



// Llama esto desde tu /status cuando la llamada termine

function notifyCallCompleted(callSid) {

  const done = PENDING.get(callSid);

  if (done) {

    PENDING.delete(callSid);

    done(); // resuelve el waiter

  } else {

    // Termin antes de armar el waiter

    COMPLETED.add(callSid);

  }

}



// ========================= Lanzar una llamada =========================

async function launchCall(contact) {

  const numero = normalizePhone(contact.telefono);



  // Evita duplicados

  if (yaFueLlamado(numero)) {

    const intentos = numerosProcesados.intentos.get(numero) || 0;

    console.log(`??  Saltando ${numero} - Ya fue llamado ${intentos} vez(es)`);

    return null;

  }



  await almacenarDatosContacto(contact);



  const payload = {

    application_sid: process.env.APPLICATION_SID,

    from: 'bot4tm',

    to: { type: 'sip', sipUri: `sip:7757${numero}@atmbponextcall.controlnextapp.com` },

    call_hook:       { url: `${process.env.WS_BASE_URL}/call`,    method: 'GET'  },

    call_status_hook:{ url: `${process.env.HTTP_BASE_URL}/status`, method: 'POST' },

    speech_synthesis_vendor:  'google',

    speech_synthesis_language:'es-US',

    speech_synthesis_voice:   'es-US-Standard-A',

    speech_recognizer_vendor: 'google',

    speech_recognizer_language:'es-CO'

  };



  try {

    console.log(`   ? Llamando a: ${numero} | Contacto: ${contact.nombrecompleto} | Deuda: ${contact.monto_inicial}`);

    const response = await client.calls.create(payload);

    const callSid = response.sid;



    marcarComoLlamado(numero);

    await guardarNumerosProcesados();



    llamadasActivas++;

    console.log(`   ? Llamada iniciada - call_sid: ${callSid}`);



    // Devuelve el "waiter" que se resuelve al cerrar la llamada

    const done = waitCallCompletion(callSid);

    return { callSid, done };

  } catch (err) {

    console.error(`   ? Error al llamar a ${numero}:`, err?.message || err);

    const status = err?.response?.status;

    const data   = err?.response?.data;

    if (status) console.error('     ? HTTP status:', status);

    if (data)   console.error('     ? Response data:', JSON.stringify(data));

    return null;

  }

}



// ========================= Procesar lote (espera cierre real) =========================

async function procesarLote(lote, numeroDeLote) {

  console.log(`\n?? PROCESANDO LOTE ${numeroDeLote} (${lote.length} contactos)`);

  console.log('-'.repeat(50));



  const waiters = [];

  let realizadas = 0;



  for (let i = 0; i < lote.length; i++) {

    if (detenerLlamadas) {

      console.log('??  Ejecucin detenida por usuario');

      break;

    }



    const contacto = lote[i];

    const res = await launchCall(contacto);

    if (res && res.done) {

      realizadas++;

      waiters.push(res.done);

    }



    // pequeo respiro para evitar rate-limits (ajusta si quieres)

    if (i < lote.length - 1) await sleep(400);

  }



  if (waiters.length > 0) {

    console.log(`? Esperando a que terminen ${waiters.length} llamada(s) del lote ${numeroDeLote}...`);

    await Promise.allSettled(waiters);

  } else {

    console.log(`??  Lote ${numeroDeLote} sin llamadas iniciadas`);

  }



  lotesCompletados++;

  console.log(`? LOTE ${numeroDeLote} COMPLETADO | Llamadas realizadas: ${realizadas}/${lote.length}`);

  console.log('-'.repeat(50));

}



// ========================= Ejecutar todas las llamadas =========================

async function ejecutarLlamadas() {

  await cargarNumerosProcesados();



  return new Promise((resolve, reject) => {

    const contactos = [];

    const lotes = [];



    console.log('?? Leyendo archivo de contactos...');

    console.log(`??  Nmeros ya procesados: ${numerosProcesados.numerosLlamados.size}`);



    fs.createReadStream(CONTACTS_FILE)

      .pipe(csv())

      .on('data', (row) => {

        if (row.telefono && String(row.telefono).trim() !== '') {

          contactos.push(row);

        }

      })

      .on('end', async () => {

        console.log(`?? Total de contactos en CSV: ${contactos.length}`);



        // Filtra los no llamados

        const contactosNuevos = contactos.filter(c =>

          !yaFueLlamado(normalizePhone(c.telefono))

        );



        console.log(`?? Contactos nuevos por llamar: ${contactosNuevos.length}`);

        if (contactosNuevos.length === 0) {

          console.log('? Nada por hacer: todos los nmeros ya fueron procesados con anterioridad');

          return resolve();

        }



        // Arma lotes de tamao BATCH_SIZE

        for (let i = 0; i < contactosNuevos.length; i += BATCH_SIZE) {

          lotes.push(contactosNuevos.slice(i, i + BATCH_SIZE));

        }



        console.log(`?? Total de lotes a procesar: ${lotes.length}`);



        // Procesa lotes secuencialmente (espera cierre real de cada lote)

        for (let i = 0; i < lotes.length; i++) {

          if (detenerLlamadas) {

            console.log('??  Ejecucin detenida por usuario');

            break;

          }

          await procesarLote(lotes[i], i + 1);



          // Guarda progreso despus de cada lote

          await guardarNumerosProcesados();



          // Pausa entre lotes (opcional)

          if (i < lotes.length - 1) {

            console.log('??  Esperando 3 segundos antes del prximo lote...\n');

            await sleep(3000);

          }

        }



        console.log('\n? TODOS LOS LOTES COMPLETADOS');

        console.log(`?? Resumen:`);

        console.log(`    Contactos en CSV:            ${contactos.length}`);

        console.log(`    Contactos nuevos procesados: ${contactosNuevos.length}`);

        console.log(`    Lotes procesados:            ${lotesCompletados}`);

        console.log(`    Nmeros procesados totales:  ${numerosProcesados.numerosLlamados.size}`);



        resolve();

      })

      .on('error', reject);

  });

}



// ========================= Estado / Control =========================

function obtenerEstadoLlamadas() {

  return {

    detenerLlamadas,

    llamadasActivas,

    lotesCompletados,

    numerosProcesados: numerosProcesados.numerosLlamados.size,

    estado: detenerLlamadas ? 'detenido' : (llamadasActivas > 0 ? 'en_ejecucion' : 'completado')

  };

}



function detenerEjecucionLlamadas() {

  detenerLlamadas = true;

  console.log('??  Se marc bandera para detener ejecucin.');

}



async function reiniciarEjecucion() {

  detenerLlamadas = false;

  llamadasActivas = 0;

  lotesCompletados = 0;

  console.log('?? Ejecucin reiniciada.');

}



// Estadsticas simples

function verEstadisticas() {

  console.log('\n?? ESTADSTICAS DE LLAMADAS');

  console.log('-'.repeat(30));

  console.log(` Nmeros procesados: ${numerosProcesados.numerosLlamados.size}`);

  console.log(` ltima ejecucin:  ${numerosProcesados.ultimaEjecucion || 'Nunca'}`);

  console.log(' Intentos por nmero (top 5):');



  const sortedIntentos = Array.from(numerosProcesados.intentos.entries())

    .sort((a, b) => b[1] - a[1])

    .slice(0, 5);



  for (const [numero, intentos] of sortedIntentos) {

    console.log(`  - ${numero}: ${intentos} intento(s)`);

  }

}



// ========================= Exports =========================

module.exports = {

  ejecutarLlamadas,

  detenerEjecucionLlamadas,

  obtenerEstadoLlamadas,

  reiniciarEjecucion,

  reiniciarHistorial,

  verEstadisticas,

  cargarNumerosProcesados,

  notifyCallCompleted // ?? llmalo desde /status cuando una llamada termine

};

