// logger.js
const pino = require('pino');
const crypto = require('crypto');

/**
 * ENV opcionales:
 *   LOG_LEVEL=info|warn|error|debug
 *   LOG_SILENCE_PATTERNS=[pat1]|[pat2]|...      (substrings o regex /.../)
 *   LOG_DEDUPE_WINDOW_MS=1000                   (ventana para suprimir repetidos)
 *   LOG_MODULE_LEVELS=yardmaster:warn,app-main:info
 *   LOG_PRETTY=1                                (activa pino-pretty en dev)
 */

const parsePatterns = (raw) => {
  if (!raw) return [];
  return raw.split('|').map(s => {
    s = s.trim();
    if (!s) return null;
    // Permite /regex/ sintaxis
    if (s.startsWith('/') && s.endsWith('/')) {
      return new RegExp(s.slice(1, -1));
    }
    // substring normal
    return s;
  }).filter(Boolean);
};

const silencePatterns = parsePatterns(process.env.LOG_SILENCE_PATTERNS);
// Ejemplo útil:
// LOG_SILENCE_PATTERNS="[AUDIO → ULTRAVOX]|[AUDIO ← ULTRAVOX]|Evento recibido en /event"

const dedupeWindowMs = Number(process.env.LOG_DEDUPE_WINDOW_MS || 1000);

// Niveles por módulo (override)
const moduleLevelMap = {};
if (process.env.LOG_MODULE_LEVELS) {
  process.env.LOG_MODULE_LEVELS.split(',').forEach(pair => {
    const [mod, lvl] = pair.split(':').map(s => s && s.trim());
    if (mod && lvl) moduleLevelMap[mod] = lvl;
  });
}

// Estado para de-dupe
const recent = new Map(); // key -> { ts, count }

// Utilidad: decidir si se silencia por patrones
function isSilenced(message) {
  if (!silencePatterns.length) return false;
  for (const p of silencePatterns) {
    if (p instanceof RegExp) {
      if (p.test(message)) return true;
    } else {
      if (message.includes(p)) return true;
    }
  }
  return false;
}

// Crea el baseLogger con hooks para filtrar/compactar
function makeBaseLogger() {
  const common = {
    level: process.env.LOG_LEVEL || 'info',
    redact: {
      paths: [
        'token',
        '*.token',
        'authorization',
        'auth',
        'headers.authorization',
        'Authorization',
      ],
      censor: '[REDACTED]'
    },
    formatters: {
      bindings: () => ({}) // sin pid/hostname
    },
    timestamp: () => `,"time":"${new Date().toISOString()}"`,
    hooks: {
      // Pino v8: intercepta cada log
      logMethod(inputArgs, method, level) {
        // Soportamos tu estilo: logger.info(ctx, message)
        // o logger.info(message, ctx)
        let msg = '';
        let ctx = {};
        if (typeof inputArgs[0] === 'string') {
          msg = inputArgs[0];
          ctx = inputArgs[1] || {};
          // normalizamos a (ctx, msg)
          inputArgs[0] = ctx;
          inputArgs[1] = msg;
        } else {
          ctx = inputArgs[0] || {};
          msg = inputArgs[1] || '';
        }

        // Silenciar por patrones
        if (typeof msg === 'string' && isSilenced(msg)) {
          return; // no loguear
        }

        // De‑dupe (mismo nivel + módulo + mensaje simple)
        const moduleName = (ctx && ctx.module) || '';
        const key = `${level}|${moduleName}|${msg}`;
        const now = Date.now();
        const rec = recent.get(key);

        if (rec && (now - rec.ts) < dedupeWindowMs) {
          // dentro de ventana: suprimir y contar
          rec.count += 1;
          rec.ts = now;
          recent.set(key, rec);
          return;
        }

        // Si había suprimidos de otra key, volcarlos ahora?:
        // Estrategia simple: cuando expira ventana de una key, el próximo log de esa misma key añade el contador.
        // Si existe rec con count > 0 y ya pasó la ventana, anexamos nota.
        if (rec && rec.count > 0 && (now - rec.ts) >= dedupeWindowMs) {
          const suppressed = rec.count;
          recent.delete(key);
          // emitimos el mensaje suprimido resumido ANTES del nuevo mensaje
          method.apply(this, [{
            ...ctx,
            deduped: true,
            suppressed
          }, `${msg} ( ${suppressed} similares suprimidos )`]);
          return;
        }

        // Registrar esta aparición
        recent.set(key, { ts: now, count: 0 });

        // Continuar con el log normal
        method.apply(this, inputArgs);
      }
    }
  };

  // pretty transport (solo si se pide)
  const usePretty = process.env.LOG_PRETTY === '1';
  if (usePretty) {
    try {
      return pino({
        ...common,
        transport: {
          target: 'pino-pretty',
          options: {
            colorize: true,
            singleLine: true,
            messageKey: 'msg',
            ignore: 'pid,hostname' // ya venimos sin bindings igualmente
          }
        }
      });
    } catch {
      // si pino-pretty no está instalado, caer al normal
      return pino(common);
    }
  }
  return pino(common);
}

const baseLogger = makeBaseLogger();

// Crea logger por módulo con override de nivel si aplica
const createLogger = (moduleName) => {
  // nivel por módulo (si se definió)
  const levelOverride = moduleLevelMap[moduleName];
  const logger = levelOverride
    ? baseLogger.child({ module: moduleName, level: levelOverride })
    : baseLogger.child({ module: moduleName });

  // API consistente con tu wrapper
  return {
    info: (message, ctx = {}) => logger.info(ctx, message),
    warn: (message, ctx = {}) => logger.warn(ctx, message),
    debug: (message, ctx = {}) => logger.debug(ctx, message),
    error: (message, ctx = {}) => {
      if (message instanceof Error) {
        logger.error({
          ...ctx,
          err: { message: message.message, stack: message.stack }
        }, 'Error ocurrido');
      } else {
        logger.error(ctx, message);
      }
    },
    child: (bindings = {}) => {
      const childName = `${moduleName}:${bindings.module || 'child'}`;
      return createLogger(childName);
    },
    pinoLogger: logger
  };
};

module.exports = createLogger;
