const convertir = require('numero-a-letras');

/**
 * Convierte valores a texto según el tipo y localización especificados
 * @param {string|number} valor - Valor a convertir
 * @param {string} tipo - Tipo de dato: 'dinero', 'telefono', 'porcentaje', 'decimal', 'entero'
 * @param {string} locale - Código de localización: 'Es-Co', 'Es-PE', 'Es-MX', 'Es-ES', etc.
 * @returns {string} Texto convertido con formato apropiado
 */
const convertirATexto = (valor, tipo = 'dinero', locale = 'Es-Co') => {
  try {
    if (valor === null || valor === undefined || valor === '') {
      return '';
    }

    // Configuración por país
    const configPaises = {
      'Es-Co': { moneda: 'pesos colombianos', codigo: 'COP' },
      'Es-PE': { moneda: 'soles peruanos', codigo: 'PEN' },
      'Es-MX': { moneda: 'pesos mexicanos', codigo: 'MXN' },
      'Es-ES': { moneda: 'euros', codigo: 'EUR' },
      'Es-AR': { moneda: 'pesos argentinos', codigo: 'ARS' },
      'Es-CL': { moneda: 'pesos chilenos', codigo: 'CLP' }
    };

    const config = configPaises[locale] || configPaises['Es-Co'];

    // Función para insertar pausas naturales
    const insertarPausas = (texto) => {
      return texto
        .replace(/\bmil\b/gi, 'mil,')
        .replace(/\b(ciento|doscientos|trescientos|cuatrocientos|quinientos|seiscientos|setecientos|ochocientos|novecientos)\b/gi, '$1,')
        .replace(/\s+con\s+/gi, ', con ')
        .replace(/\s+/g, ' ')
        .trim();
    };

    // Función para pronunciar dígitos individualmente
    const pronunciarDigitos = (numeroStr) => {
      const digitos = String(numeroStr).split('');
      const pronunciacion = digitos.map(digito => {
        const digitosTexto = {
          '0': 'cero', '1': 'uno', '2': 'dos', '3': 'tres', '4': 'cuatro',
          '5': 'cinco', '6': 'seis', '7': 'siete', '8': 'ocho', '9': 'nueve'
        };
        return digitosTexto[digito] || digito;
      });
      return pronunciacion.join(', ');
    };

    // Convertir valor a número limpio
    const limpiarNumero = (val) => {
      if (typeof val === 'number') return val;
      
      const str = String(val).replace(/[^0-9.,]/g, '').replace(',', '.');
      const numero = parseFloat(str);
      return isNaN(numero) ? 0 : numero;
    };

    // Procesar según el tipo
    switch (tipo.toLowerCase()) {
      case 'dinero':
        const monto = limpiarNumero(valor);
        let textoMoneda = convertir.NumerosALetras(monto);
        
        // Ajustar según la moneda del país
        textoMoneda = textoMoneda.replace(/\s*\d+\/100\s*(M\.N\.)?/gi, '');
        textoMoneda = textoMoneda.replace(/Pesos?/gi, config.moneda);
        
        return insertarPausas(textoMoneda);

      case 'telefono':
        const telefono = String(valor).replace(/\D/g, '');
        return pronunciarDigitos(telefono);

      case 'porcentaje':
        const porcentaje = limpiarNumero(valor);
        let textoPorcentaje = convertir.NumerosALetras(porcentaje);
        textoPorcentaje = textoPorcentaje.replace(/\s*\d+\/100\s*/gi, '');
        return `${insertarPausas(textoPorcentaje)} por ciento`;

      case 'decimal':
        const decimal = limpiarNumero(valor);
        let textoDecimal = convertir.NumerosALetras(decimal);
        textoDecimal = textoDecimal.replace(/\s*\d+\/100\s*/gi, '');
        return insertarPausas(textoDecimal);

      case 'entero':
        const entero = Math.round(limpiarNumero(valor));
        let textoEntero = convertir.NumerosALetras(entero);
        textoEntero = textoEntero.replace(/\s*\d+\/100\s*/gi, '');
        return insertarPausas(textoEntero);

      case 'fecha':
        // Para fechas en formato DD/MM/YYYY o YYYY-MM-DD
        const fecha = new Date(valor);
        if (!isNaN(fecha.getTime())) {
          const dia = fecha.getDate();
          const mes = fecha.toLocaleString('es-ES', { month: 'long' });
          const año = fecha.getFullYear();
          return `${convertir.NumerosALetras(dia)} de ${mes} de ${convertir.NumerosALetras(año)}`;
        }
        return String(valor);

      case 'cantidad':
        const cantidad = Math.round(limpiarNumero(valor));
        let textoCantidad = convertir.NumerosALetras(cantidad);
        return insertarPausas(textoCantidad);

      default:
        return String(valor);
    }
  } catch (error) {
    console.error('Error en convertirATexto:', error);
    return String(valor);
  }
};

// Mantener compatibilidad con el código existente
const convertirNumeroATexto = (raw, tipo = 'dinero', locale = 'Es-Co') => {
  return convertirATexto(raw, tipo, locale);
};

// Ejemplos de uso:
/*
console.log(convertirATexto(900, 'dinero', 'Es-Co')); 
// "novecientos pesos colombianos"

console.log(convertirATexto(900, 'dinero', 'Es-PE')); 
// "novecientos soles peruanos"

console.log(convertirATexto('958620970', 'telefono', 'Es-PE')); 
// "nueve, cinco, ocho, seis, dos, cero, nueve, siete, cero"

console.log(convertirATexto(25.5, 'porcentaje', 'Es-Co')); 
// "veinticinco punto cinco por ciento"

console.log(convertirATexto(1234.56, 'decimal', 'Es-MX')); 
// "mil doscientos treinta y cuatro punto cincuenta y seis"

console.log(convertirATexto(1500, 'cantidad', 'Es-Co')); 
// "mil quinientos"
*/

module.exports = convertirNumeroATexto;

/*const convertir = require('numero-a-letras');

/*function convertirNumeroATexto(transcripts = []) {
  const texto = transcripts
    .filter(t => t.role === 'agent' && t.final)
    .map(t => t.text)
    .filter(Boolean)
    .join(' ');

  return texto.replace(/(?:\$|\b)?(\d{1,3}(?:[\.,]?\d{3})*(?:[\.,]\d+)?|\d+)(?:%|h|\/\d+)?\b/g, (match) => {
    const limpio = match.replace(/[^\d.,]/g, ''); // Elimina % $ h
    const decimal = limpio.replace(',', '.').replace(/(?<=\.\d*)\./g, ''); // 1.234,56 → 1234.56
    const numero = parseFloat(decimal);

    if (isNaN(numero)) return match;

    const textoNumero = numeroALetras(numero);

    // Reconstruir contexto si era porcentaje o moneda
    if (match.includes('%')) return `${textoNumero} por ciento`;
    if (match.includes('$')) return `${textoNumero} pesos`;
    if (/\/\d+/.test(match)) return `${textoNumero} dividido por ${match.split('/')[1]}`;
    if (match.endsWith('h')) return `${textoNumero} horas`;

    return textoNumero;
  });
}*/


/*function convertirNumeroATexto(numero) {
  try {
    return numeroALetras.numeroALetras(numero, {
      plural: 'pesos',
      singular: 'peso',
      centPlural: 'centavos',
      centSingular: 'centavo'
    });
  } catch (e) {
    return `${numero}`; // fallback si falla
  }
}*/



/*const convertirNumeroATexto = (raw) => {
  try {
    if (typeof raw !== 'string') raw = String(raw);

    const limpiarTexto = (texto) => {
      return texto
        .replace(/\s*\d+\/100\s*(M\.N\.)?/gi, '')              
        .replace(/Pesos?/gi, 'pesos colombianos')              
        .replace(/\s{2,}/g, ' ')                                
        .trim();
    };

    // Porcentaje
    if (raw.includes('%')) {
      const valor = parseFloat(raw.replace('%', '').replace(',', '.'));
      if (!isNaN(valor)) {
        let texto = convertir.NumerosALetras(valor);
        return `${insertarPausas(texto)} por ciento`;
      }
    }

    // Con símbolo de pesos
    if (raw.includes('$')) {
      const valor = parseFloat(raw.replace(/[^0-9.,]/g, '').replace(',', '.'));
      if (!isNaN(valor)) {
        let texto = convertir.NumerosALetras(valor);
        texto = limpiarTexto(texto);
        return insertarPausas(texto);
      }
    }

    // Número genérico
    const valor = parseFloat(raw.replace(/[^0-9.,]/g, '').replace(',', '.'));
    if (!isNaN(valor)) {
      let texto = convertir.NumerosALetras(valor);
      texto = limpiarTexto(texto);
      return insertarPausas(texto);
    }

    return raw;
  } catch (e) {
    return String(raw);
  }
};



const insertarPausas = (texto) => {
  return texto
   // .replace(/\d{1,3}(?=(\d{3})+(?!\d))/g, '$&,')
    .replace(/\bmil\b/gi, 'mil,')
    .replace(/\b(ciento|doscientos|trescientos|cuatrocientos|quinientos|seiscientos|setecientos|ochocientos|novecientos)\b/gi, '$1,')
    //.replace(/\b y\b/gi, ', y')
    .replace(/\s+con\s+/gi, ', con ')
    .replace(/\s+/g, ' ')
    .trim();
};












module.exports = convertirNumeroATexto;*/