// utils/transcript-manager.js
const { pool } = require('../db/pg');

class TranscriptManager {
  // Inserta un nico turno
  static async saveMessage(callSid, { speaker = 'BOT', message = '', timestamp, meta } = {}) {
    if (!callSid || !message?.trim()) return false;

    try {
      if (timestamp && meta !== undefined) {
        await pool.query(
          'SELECT telephony.insert_transcript($1,$2,$3,$4,$5)',
          [callSid, speaker, message, timestamp, meta]
        );
      } else if (timestamp) {
        await pool.query(
          'SELECT telephony.insert_transcript($1,$2,$3,$4)',
          [callSid, speaker, message, timestamp]
        );
      } else if (meta !== undefined) {
        await pool.query(
          'SELECT telephony.insert_transcript($1,$2,$3, DEFAULT, $4)',
          [callSid, speaker, message, meta]
        );
      } else {
        await pool.query(
          'SELECT telephony.insert_transcript($1,$2,$3)',
          [callSid, speaker, message]
        );
      }
      return true;
    } catch (err) {
      console.error('Error saving message to database:', err);
      return false;
    }
  }

  // Inserta una transcripcin (array o objeto)
  static async saveTranscript(callSid, transcriptData) {
    try {
      if (Array.isArray(transcriptData)) {
        for (const t of transcriptData) {
          const message = (t?.text ?? t?.message ?? '').trim();
          if (!message) continue;
          await this.saveMessage(callSid, {
            speaker: (t?.role || 'USER').toUpperCase(),
            message
          });
        }
        return true;
      }

      if (transcriptData && typeof transcriptData === 'object') {
        const message = (transcriptData.text ?? transcriptData.message ?? '').trim();
        if (!message) return false;
        return this.saveMessage(callSid, {
          speaker: (transcriptData.role || 'USER').toUpperCase(),
          message
        });
      }

      return false;
    } catch (err) {
      console.error('Error saving transcript:', err);
      return false;
    }
  }

  // Obtiene transcripcin completa (ajusta columna segn tu funcin SQL)
  static async getFullTranscript(callSid) {
    try {
      // si tu funcin devuelve columna 'ts', mejor ordenar por 'ts'
      const result = await pool.query(
        `SELECT * FROM telephony.get_call_transcript($1) ORDER BY ts`,
        [callSid]
      );
      return result.rows;
    } catch (error) {
      console.error('Error getting transcript:', error);
      return [];
    }
  }

  static async getCallSummary(callSid) {
    try {
      const result = await pool.query(
        `SELECT telephony.get_call_summary($1) AS summary`,
        [callSid]
      );
      return result.rows[0]?.summary || {};
    } catch (error) {
      console.error('Error getting call summary:', error);
      return {};
    }
  }

  static formatTranscriptForExport(transcript) {
    return transcript.map((turn) => {
      // intenta ambas por si tu funcin retorna 'ts' en vez de 'timestamp'
      const ts = turn.timestamp || turn.ts;
      const time = ts ? new Date(ts).toLocaleTimeString('es-PE', { hour12: false }) : '--:--:--';
      return `[${time}] ${turn.speaker}: ${turn.message}`;
    }).join('\n');
  }
}

module.exports = TranscriptManager;
