const Emitter = require('events');
const WebSocket = require('ws');
const fs = require('fs');
const path = require('path');
const os = require('os');
const axios = require('axios');
// Static counter for all instances of Adaptor class
let instanceCounter = 0;


class Adaptor extends Emitter {
  constructor(logger, api_key, prompt, model = 'fixie-ai/ultravox', voice = 'Andrea-Spanish') {
    super();
    this.logger = logger;
    this.api_key = api_key;
    this.prompt = prompt;
    this.model = model;
    this.voice = voice;
    this.joinUrl = null;
    this.initialized = false;
    this.errored = false;

    this.outgoingBufferQueue = []; // audio desde Ultravox pendiente → Jambonz OUT

    
    this.bytesToUltravox = 0;
    this.bytesFromUltravox = 0;



    this.enableFileLogging = process.env.DEBUG_AUDIO_FILE === 'true';
    if (this.enableFileLogging) {
      // Assign unique instance ID for file naming
      this.instanceId = ++instanceCounter;
      const recordingsPath = path.resolve(__dirname, '..', 'recordings');
      
      if (!fs.existsSync(recordingsPath)) fs.mkdirSync(recordingsPath);
      
      // Create temp file paths for storing incoming and outgoing audio for debugging
      this.incomingAudioFilePath = path.join(recordingsPath, `jambonz-in-audio-ultravox-${this.instanceId}.raw`);
      this.outgoingAudioFilePath = path.join(recordingsPath, `ultravox-out-audio-${this.instanceId}.raw`);
      // Clear the files
      fs.writeFileSync(this.incomingAudioFilePath, Buffer.alloc(0));
      fs.writeFileSync(this.outgoingAudioFilePath, Buffer.alloc(0));
      this.logger.info(`Audio logs: IN→ ${this.incomingAudioFilePath}, OUT← ${this.outgoingAudioFilePath}`);

    }

    // Add default error handler to prevent crashes
    this.on('error', (err) => {
      this.errored = true;
      this.logger.error({err: err.message}, 'Ultravox adapter error caught by default handler');
    });

    // Start the connection process
    this._safeInitialize();
  }

  async _safeInitialize() {
    try {
      await this._initializeConnection();
    } catch (err) {
      this.errored = true;
      this.logger.error({err: err.message, stack: err.stack}, 'Failed in _safeInitialize');
      // Do not emit or throw errors here
    }
  }

  async _initializeConnection() {
  
      // First create a call to get the joinUrl
      /*const callData = await this._createCall();

      // Check if there was an error during call creation
      if (callData.error || !callData.joinUrl){
        this.errored = true;
        this.logger.error({ message: callData.message }, 'Failed to create Ultravox call');
        return; // Don't emit error, just log and return
      }

      this.joinUrl = callData.joinUrl;*/
      this._connect();
      this.initialized = true;
  }


  async _createCall() {
    const payload = {
      systemPrompt: 'Eres una agente llamada Andrea...',
      model: 'fixie-ai/ultravox-70B',
      voice: 'Andrea-Spanish',
      transcriptOptional: true,
      initialOutputMedium: 'MESSAGE_MEDIUM_VOICE',
      firstSpeakerSettings: {
        agent: {
          prompt: 'Hola, me llamo Andrea del área de cobranza de la empresa Claro. ¿Podrías brindarme un momento para conversar sobre tu deuda pendiente?',
          uninterruptible: false
        }
      },
      inactivityMessages: [{
        duration: '120s',
        message: '¿Sigues ahí?',
        endBehavior: 'END_BEHAVIOR_HANG_UP_SOFT'
      }],
      medium: {
        serverWebSocket: {
          inputSampleRate: 8000,
          outputSampleRate: 8000,
          clientBufferSizeMs: 300
        }
      }
    };

    try {
      const response = await axios.post('https://api.ultravox.ai/api/calls', payload, {
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': this.api_key
        }
      });

      if (response.status !== 201 || !response.data?.joinUrl) {
        return { error: true, message: 'Invalid response from Ultravox' };
      }

      this.logger.info({ joinUrl: response.data.joinUrl }, 'Ultravox call created');
      return response.data;
    } catch (err) {
      const status = err.response?.status;
      const detail = err.response?.data?.detail;
      if (status === 402) {
        return { error: true, message: `Ultravox subscription required: ${detail}` };
      }

      return { error: true, message: err.message };
    }
  }

  setOutgoingJambonzSocket(ws) {
    this.ws_jambonz_out = ws;
    this.logger.info('[SETUP] ws_jambonz_out ha sido configurado');

   if (this.ws_jambonz_out?.readyState === WebSocket.OPEN && this.outgoingBufferQueue.length > 0) {
     this.logger.info(`[ULTRAVOX] Drenando ${this.outgoingBufferQueue.length} chunks pendientes hacia Jambonz`);
    try {
       for (const chunk of this.outgoingBufferQueue) this.ws_jambonz_out.send(chunk);
     } catch (e) {
       this.logger.error({ e }, 'Error drenando cola hacia Jambonz');
      } finally {
        this.outgoingBufferQueue = [];
      }
   }
  }
  
  setIncomingJambonzSocket(ws) {

    if (this.ws_jambonz_in && this.ws_jambonz_in !== ws) {
    this.logger.warn('[UltravoxAdapter] ws_jambonz_in ya estaba asignado, cerrando anterior');
    try { this.ws_jambonz_in.close(); } catch (e) {}
    }

  this.ws_jambonz_in = ws;

    if (this.errored) {
      this.logger.warn('Not setting up incoming socket - adapter is in error state');
      return;
    }
  
    this.incomingBufferQueue = [];
    this.ws_jambonz_in = ws;

    ws.on('message', (message, isBinary) => {
      if (!isBinary) return;
      this.logger.debug(`[AUDIO → ULTRAVOX] ${message.length} bytes`);
      this.bytesToUltravox += message.length;

      if (this.enableFileLogging) {
        fs.appendFileSync(this.incomingAudioFilePath, message);
      }

      if (this.ws && this.ws.readyState === WebSocket.OPEN) {
        this.ws.send(message);
      } else {
        this.logger.warn('[ULTRAVOX] No conectado aún. Encolando audio.');
        this.incomingBufferQueue.push(message);
      }
    });
    ws.on('close', () => {
      this.logger.info('Jambonz input socket cerrado');
      this.close();
    });

    ws.on('error', (err) => {
      this.logger.error({ err }, 'Error en socket entrante de Jambonz');
      this.close();
    });
  
  }
  
 /* setIncomingJambonzSocket(ws) {
    // Check if we're in an error state before setting up socket
    if (this.errored) {
      this.logger.warn('Not setting up incoming socket - adapter is in error state');
      return;
    }

    ws
      .on('message', (message, isBinary) => {
        // send audio data to Ultravox
        console.log('[WS] Mensaje recibido', { isBinary, length: message?.length });
        if (isBinary && this.ws &&  WebSocket.OPEN === 1) {
          //const json = JSON.parse(message.toString());
          console.log("data binaria", message);
          if (this.enableFileLogging) {
            fs.appendFileSync(this.incomingAudioFilePath, message);
            this.logger.debug(`Wrote ${message.length} bytes from Jambonz to ${this.incomingAudioFilePath}`);
          }

          // No need to base64 encode for Ultravox - send raw PCM audio
          this.ws.send(message);
        }
      })
      .on('close', () => {
        this.logger.info('Jambonz socket closed');
     
        this.close();
      })
      .on('error', (err) => {
        this.logger.info({ err }, 'Jambonz socket error');
        this.close();
      });
  }*/

 

      close() {
        this.logger.info('Cerrando conexión a Ultravox');
        this.ws?.close();
        if (this.enableFileLogging) {
          this.logger.info(`Audio grabado IN→ ${this.incomingAudioFilePath}`);
          this.logger.info(`Audio grabado OUT← ${this.outgoingAudioFilePath}`);
        }
      }
  _connect() {
    if (!this.joinUrl) {
      this.logger.error('Cannot connect to Ultravox: No joinUrl available');
      return;
    }

    this.logger.info({ joinUrl: this.joinUrl }, '2 Connecting to Ultravox WebSocket');
    try {

      
      this.ws = new WebSocket(this.joinUrl);
      console.log("open!,",this.ws);
      this.ws
        .on('open', this._onOpen.bind(this))
        .on('error', this._onError.bind(this))
        .on('close', this._onClose.bind(this))
        .on('message', this._onServerEvent.bind(this));
    } catch (err) {
      this.logger.error({ err }, 'Error creating WebSocket connection');
      // Don't throw or emit, just log
    }
  }

  _onOpen() {
    this.logger.info('Ultravox WebSocket connection open');
    this.emit('ready');
  
    // 🔁 Si hay mensajes en cola, enviarlos ahora
    if (this.incomingBufferQueue?.length > 0) {
      this.logger.info(`Enviando ${this.incomingBufferQueue.length} mensajes del buffer`);
      this.incomingBufferQueue.forEach(buf => {
        try {
          this.ws.send(buf);
        } catch (e) {
          this.logger.error({ e }, 'Error al enviar buffer tras apertura');
        }
      });
      this.incomingBufferQueue = [];
    }
  }
  
 /* _onOpen() {
    this.logger.info('Ultravox WebSocket connection open');
    this.emit('ready');
    // No need to send an initial message - already configured in createCall
  }*/

  _onClose(code, reason) {
    this.logger.info({ code, reason }, 'Ultravox disconnected');
    // Don't emit close, just log
  }

  _onError(err) {
    this.logger.info({ err }, 'Ultravox WebSocket error');
    // Don't emit error, just log
  }

  _onMessage(data, isBinary) {
    this.logger.debug(isBinary ? '[binary message received]' : data.toString())
  }

  _onServerEvent(message, isBinary) {
    if (isBinary) {
      this.bytesFromUltravox += message.length;
      this.logger.info(`[AUDIO ← ULTRAVOX] ${message.length} bytes`);
  
      this._sendToJambonz(message);
  
      if (this.enableFileLogging) {
        fs.appendFileSync(this.outgoingAudioFilePath, message);
      }
  
    } else {
      try {
        const data = JSON.parse(message);
        this.logger.debug({ data }, 'Mensaje de control de Ultravox');
      } catch (err) {
        this.logger.error({ err, message }, 'Error parsing mensaje Ultravox');
      }
    }
  }

 /* _onServerEvent(message, isBinary) {
    if (isBinary) {
      this.bytesFromUltravox += message.length;
      this.logger.info(`[AUDIO ← ULTRAVOX] ${message.length} bytes`);
      this._sendToJambonz(message);
      if (this.enableFileLogging) {
        fs.appendFileSync(this.outgoingAudioFilePath, message);
      }

      if (this.ws_jambonz_out && this.ws_jambonz_out.readyState === WebSocket.OPEN) {
        this.ws_jambonz_out.send(message);
        this.logger.info(`[AUDIO ← ULTRAVOX] ${message.length} bytes enviados a Jambonz`);
      } else {
        this.logger.warn('[JAMBONZ] no está listo para recibir audio de Ultravox');
        this.logger.debug('readyState actual:', this.ws_jambonz_out?.readyState);
      }
    } else {
      try {
        const data = JSON.parse(message);
        this.logger.debug({ data }, 'Mensaje de control de Ultravox');
      } catch (err) {
        this.logger.error({ err, message }, 'Error parsing mensaje Ultravox');
      }
    }
  }*/

  _sendToJambonz(audioChunk) {
    if (this.ws_jambonz_out && this.ws_jambonz_out.readyState === WebSocket.OPEN) {
      this.ws_jambonz_out.send(audioChunk);
      this.logger.info(`[AUDIO ← ULTRAVOX] ${audioChunk.length} bytes enviados a Jambonz`);
    } else {
     // this.logger.warn('[JAMBONZ] no está listo para recibir audio de Ultravox');
      //this.logger.debug('readyState actual:', this.ws_jambonz_out?.readyState);
     this.outgoingBufferQueue.push(audioChunk);
     if (this.outgoingBufferQueue.length % 50 === 1) {
     this.logger.warn(`[JAMBONZ OUT] no listo; encolados=${this.outgoingBufferQueue.length}`);
      this.logger.debug('readyState actual:', this.ws_jambonz_out?.readyState);
     }
    }
  }

  // Public method to check if adapter is in a working state
  isHealthy() {
    return this.initialized && !this.errored && this.ws?.readyState === WebSocket.OPEN;
  }
}

module.exports = Adaptor;
