const { pool } = require('./pg');

async function saveEvent(payload) {
  // Llama a la funcin PL/pgSQL (toda la lgica en BD)
  await pool.query('SELECT telephony.ingest_event($1::jsonb)', [payload]);
}

async function todayNumbers(tz = 'America/Lima') {
  const { rows } = await pool.query('SELECT telefono FROM telephony.today_numbers($1)', [tz]);
  return rows.map(r => r.telefono);
}

async function getCallsAndTranscriptsToday(tz = 'America/Lima') {
  // 1) Calls de hoy
  const callsSql = `
    with bounds as (
      select
        date_trunc('day', (now() at time zone $1)) at time zone $1 as day_start,
        (date_trunc('day', (now() at time zone $1)) + interval '1 day') at time zone $1 as day_end
    )
    select
      c.call_sid::text,
      c.telefono,
      coalesce(c.identificacion,'') as identificacion,
      c.trunk,
      c.final_status,
      c.final_sip_status,
      c.final_sip_reason,
      (c.started_at at time zone $1) as started_at,
      (c.answered_at at time zone $1) as answered_at,
      (c.ended_at   at time zone $1) as ended_at,
      extract(epoch from (c.ended_at - c.answered_at))::int as duration_s
    from telephony.calls c
    join bounds b on true
    where (c.started_at at time zone $1) >= b.day_start
      and (c.started_at at time zone $1) <  b.day_end
    order by c.started_at asc
  `;

  const { rows: calls } = await pool.query(callsSql, [tz]);

  // 2) Transcripts de esas calls
  const sids = calls.map(c => c.call_sid);
  let transcripts = [];
  if (sids.length) {
    const txSql = `
      select
        t.call_sid::text,
        (t.ts at time zone $2) as ts_local,
        coalesce(t.speaker,'') as speaker,
        coalesce(t.message,'') as message
      from telephony.call_transcripts t
      where t.call_sid = any($1::uuid[])
      order by t.call_sid, t.ts
    `;
    const { rows } = await pool.query(txSql, [sids, tz]);
    transcripts = rows;
  }

  return { calls, transcripts };
}

module.exports = { saveEvent, todayNumbers,getCallsAndTranscriptsToday };
