// routes/http/auth.js
const express = require('express');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const router = express.Router();
const { pool } = require('../../db/pg'); // Ajusta segn tu configuracin de DB

const JWT_SECRET = process.env.JWT_SECRET || 'tu_jwt_secret_seguro';

router.use(express.json());

// Login de usuario
router.post('/login', async (req, res) => {
  try {
  
  
    const { email, password } = req.body;

    if (!email || !password) {
      return res.status(400).json({ 
        success: false, 
        message: 'Email y contrasea son requeridos' 
      });
    }

    // Buscar usuario por email
    const userQuery = `
      SELECT u.*, c.id as company_id, c.name as company_name, c.rut
      FROM telephony.users u
      INNER JOIN telephony.companies c ON u.company_id = c.id
      WHERE u.email = $1 AND u.is_active = true AND c.is_active = true
    `;

    const userResult = await pool.query(userQuery, [email.toLowerCase()]);
    
    if (userResult.rows.length === 0) {
      return res.status(401).json({ 
        success: false, 
        message: 'Credenciales invlidas' 
      });
    }

    const user = userResult.rows[0];

    // Verificar contrasea
    const isValidPassword = await bcrypt.compare(password, user.password_hash);
    if (!isValidPassword) {
      return res.status(401).json({ 
        success: false, 
        message: 'Credenciales invlidas' 
      });
    }

    // Obtener configuracin de la empresa
    const configQuery = `
      SELECT config_key, config_value, config_type, is_encrypted
      FROM telephony.master_config 
      WHERE company_id = $1 AND is_active = true
    `;

    const configResult = await pool.query(configQuery, [user.company_id]);
    
    const config = {};
    configResult.rows.forEach(row => {
      config[row.config_key] = {
        value: row.config_value,
        type: row.config_type,
        encrypted: row.is_encrypted
      };
    });

    // Generar token JWT
    const token = jwt.sign(
      { 
        userId: user.id, 
        companyId: user.company_id,
        email: user.email,
        role: user.role
      },
      JWT_SECRET,
      { expiresIn: '24h' }
    );

    // Actualizar ltimo login
    await pool.query(
      'UPDATE telephony.users SET last_login = NOW() WHERE id = $1',
      [user.id]
    );

    // Respuesta exitosa
    res.json({
      success: true,
      message: 'Login exitoso',
      data: {
        user: {
          id: user.id,
          email: user.email,
          full_name: user.full_name,
          role: user.role,
          company: {
            id: user.company_id,
            name: user.company_name,
            rut: user.rut
          }
        },
        config: config,
        token: token
      }
    });

  } catch (error) {
    console.error('Error en login:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor' ,
      error: error.message 
    });
  }
});

// Verificar token (middleware)
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN

  if (!token) {
    return res.status(401).json({ 
      success: false, 
      message: 'Token de acceso requerido' 
    });
  }

  jwt.verify(token, JWT_SECRET, (err, user) => {
    if (err) {
      return res.status(403).json({ 
        success: false, 
        message: 'Token invlido o expirado' , 
        error: error.message 
      });
    }
    req.user = user;
    next();
  });
};

// Ruta para verificar token y obtener datos actualizados
router.get('/verify', authenticateToken, async (req, res) => {
  try {
    const userQuery = `
      SELECT u.*, c.id as company_id, c.name as company_name, c.rut
      FROM telephony.users u
      INNER JOIN telephony.companies c ON u.company_id = c.id
      WHERE u.id = $1 AND u.is_active = true AND c.is_active = true
    `;

    const userResult = await pool.query(userQuery, [req.user.userId]);
    
    if (userResult.rows.length === 0) {
      return res.status(404).json({ 
        success: false, 
        message: 'Usuario no encontrado' 
      });
    }

    const user = userResult.rows[0];

    // Obtener configuracin actualizada
    const configQuery = `
      SELECT config_key, config_value, config_type, is_encrypted
      FROM telephony.master_config 
      WHERE company_id = $1 AND is_active = true
    `;

    const configResult = await pool.query(configQuery, [user.company_id]);
    
    const config = {};
    configResult.rows.forEach(row => {
      config[row.config_key] = {
        value: row.config_value,
        type: row.config_type,
        encrypted: row.is_encrypted
      };
    });

    res.json({
      success: true,
      data: {
        user: {
          id: user.id,
          email: user.email,
          full_name: user.full_name,
          role: user.role,
          company: {
            id: user.company_id,
            name: user.company_name,
            rut: user.rut
          }
        },
        config: config
      }
    });

  } catch (error) {
    console.error('Error verificando token:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Error interno del servidor' 
    });
  }
});

module.exports = { authRouter: router, authenticateToken };