// routes/transfer.js
const express = require('express');
const jambonz = require('@jambonz/node-client');
const { WebhookResponse } = jambonz;
const { publish } = require('../../utils/monitor-hub');

const router = express.Router();
const pendingTransfers = new Map();

function getJambonzClient() {
  const accountSid = process.env.JAMBONZ_ACCOUNT_SID || process.env.JAMBONZ_ACCOUNT;
  const apiKey = process.env.JAMBONZ_API_KEY || process.env.JAMBONZ_API;
  const baseUrl = process.env.JAMBONZ_BASE_URL || process.env.JAMBONZ_BASE;
  
  if (!accountSid || !apiKey) {
    throw new Error('Faltan credenciales JAMBONZ en env');
  }
  return jambonz(accountSid, apiKey, { baseUrl });
}

router.post('/', async (req, res) => {
  const logger = req.app?.locals?.logger || console;
  const body = req.body || {};
  const call_sid = body.call_sid || body.callSid;
  const referTo = body.referTo || process.env.HUMAN_AGENT_NUMBER;

  console.log('=== TRANSFER DEBUG ===');
  console.log('call_sid:', call_sid);
  console.log('referTo:', referTo);
  console.log('JAMBONZ_ACCOUNT_SID:', process.env.JAMBONZ_ACCOUNT_SID);
  console.log('JAMBONZ_BASE_URL:', process.env.JAMBONZ_BASE_URL);
  console.log('HTTP_BASE_URL:', process.env.HTTP_BASE_URL);
  console.log('=====================');

  if (!call_sid) {
    return res.status(400).json({ error: 'missing call_sid' });
  }
  if (!referTo) {
    return res.status(400).json({ 
      error: 'missing referTo and no HUMAN_AGENT_NUMBER in env' 
    });
  }

  try {
    const client = getJambonzClient();
    console.log('? Cliente Jambonz creado');
    
    const callHookUrl = `${process.env.HTTP_BASE_URL}/call-hook`;
    console.log('?? Call hook URL:', callHookUrl);
    

    pendingTransfers.set(call_sid, { referTo, timestamp: Date.now() });
    
   
    console.log('?? Actualizando call_hook...');
    const updateResult = await client.calls.update(call_sid, {
      call_hook: {
        url: callHookUrl,
        method: 'POST'
      }
    });
    
    console.log('? Call hook actualizado:', updateResult);
    
    res.status(200).json({ 
      message: 'transfer initiated', 
      call_sid, 
      target: referTo 
    });
  } catch (err) {
    console.error('? Error en transferencia:', err);
    console.error('? Error details:', err.response?.data || err.message);
    pendingTransfers.delete(call_sid);
    res.status(500).json({ error: err.message });
  }
});

router.post('/call-hook', (req, res) => {
  const logger = req.app?.locals?.logger || console;
  const body = req.body || {};
  const call_sid = body.call_sid || body.callSid;
  
  logger.info({ body }, 'call-hook invoked for transfer');
  

  const transferInfo = pendingTransfers.get(call_sid);
  const referTo = transferInfo?.referTo || process.env.HUMAN_AGENT_NUMBER;

  publish({ 
    type: 'transfer', 
    call_sid, 
    status: 'callhook_received', 
    target: referTo, 
    ts: Date.now() 
  });

  const app = new WebhookResponse();
  

  pendingTransfers.delete(call_sid);
  
  app
    .pause({ length: 0.5 })
    .say({ 
      text: 'Por favor espere mientras le conecto con un agente.' 
    })
    .dial({
      actionHook: '/dial-action',
      callerId: process.env.CALLER_ID_OVERRIDE || "bot4tm",
      target: [
        {
          type: 'sip', 
          sipUri: `sip:${referTo}@${process.env.SIP_TRUNK_DOMAIN}`
        }
      ]
    });

  res.status(200).json(app);
});

router.post('/dial-action', (req, res) => {
  const logger = req.app?.locals?.logger || console;
  const body = req.body || {};
  const call_sid = body.call_sid || body.callSid;
  
  logger.info({ body }, 'dial-action called');
  
  publish({ 
    type: 'transfer', 
    call_sid, 
    status: 'dial_finished', 
    ts: Date.now() 
  });

  const app = new WebhookResponse();
  

  app
    .say({ 
      text: 'La llamada con el agente ha finalizado. Gracias.' 
    })
    .hangup();  
    
  res.status(200).json(app);
});


setInterval(() => {
  const now = Date.now();
  const timeout = 5 * 60 * 1000; 
  for (const [call_sid, info] of pendingTransfers.entries()) {
    if (now - info.timestamp > timeout) {
      pendingTransfers.delete(call_sid);
    }
  }
}, 60000);

module.exports = router;