// routes/http/upload.js
const express = require('express');
const multer  = require('multer');
const fs = require('fs');
const fsp = fs.promises;
const path = require('path');
const xlsx = require('xlsx');

const router = express.Router();
const upload = multer({ dest: 'uploads/', limits: { fileSize: 10 * 1024 * 1024 } });

const DEST_DIR = path.join(__dirname, '../../data');
const DEST_CSV = path.join(DEST_DIR, 'contactos.csv');

router.post('/', upload.single('archivo'), async (req, res) => {
  try {
    const f = req.file;
    if (!f) return res.status(400).json({ error: 'Archivo no subido' });

    const ext = path.extname(f.originalname).toLowerCase();
    await fsp.mkdir(DEST_DIR, { recursive: true });

    if (ext === '.csv') {
      // mover directo
      await fsp.rename(f.path, DEST_CSV);
    } else if (ext === '.xlsx' || ext === '.xls') {
      // convertir a CSV con columnas: telefono|identificacion
      const wb = xlsx.readFile(f.path);
      const ws = wb.Sheets[wb.SheetNames[0]];
      const rows = xlsx.utils.sheet_to_json(ws, { defval: '', raw: false });

      const norm = s => s.toString().toLowerCase()
        .normalize('NFD').replace(/[\u0300-\u036f]/g, '').trim();

      const out = ['telefono|identificacion'];
      for (const r of rows) {
        let telefono = '';
        let identificacion = '';

        for (const [k, v] of Object.entries(r)) {
          const nk = norm(k);
          if (!telefono && /(telefono|numero|celular|phone)/.test(nk)) {
            telefono = String(v).replace(/[^\d+]/g, '').trim();
          }
          if (!identificacion && /(identificacion|identificador|documento|dni|id)/.test(nk)) {
            identificacion = String(v ?? '').toString().trim();
          }
        }
        if (telefono) {
          out.push(`${telefono}|${identificacion || ''}`);
        }
      }

      await fsp.writeFile(DEST_CSV, out.join('\n'), 'utf8');
      await fsp.unlink(f.path);
    } else {
      await fsp.unlink(f.path);
      return res.status(400).json({ error: 'Formato no soportado. Use .csv, .xlsx o .xls' });
    }

    // NO se llama ejecutarLlamadas aquí. Tu call-launcher/scheduler lo detecta.
    return res.json({
      message: 'Archivo cargado y preparado',
      destino: 'data/contactos.csv'
    });
  } catch (err) {
    try { if (req.file?.path) await fsp.unlink(req.file.path); } catch {}
    console.error('Error upload:', err);
    return res.status(500).json({ error: 'Error al procesar archivo', details: err.message });
  }
});

module.exports = router;
