// file: routes/ws/call-handler.js
const { detenerEjecucionLlamadas } = require('../../utils/call-launcher');
const safeClose = require('../../utils/safe-close');
const yardMaster = require('../../utils/yardmaster');
const { WebhookResponse } = require('@jambonz/node-client');
const Adaptor = require('../../utils/ultravox_s2s');
const TranscriptManager = require('../../utils/transcript-manager');
const storage = require('../../utils/storage');
const convertirNumeroATexto = require('../../utils/numero-a-texto');

// ========================= DETECCIN DE BUZN =========================
const detectarBuzonVoz = (transcripcion) => {
  if (!transcripcion) return false;
  
  const palabrasBuzon = [
    'secretaria electronica',
    'secretara electrnica',
    'buzn de voz',
    'buzon de voz',
    'voice mail',
    'mailbox',
    'deje su mensaje',
    'deje un mensaje',
    'after the tone',
    'mensaje despus del tono',
    'leave a message',
    'Servicio de contestador'
  ];
  
  const texto = transcripcion.toLowerCase();
  return palabrasBuzon.some(palabra => texto.includes(palabra));
};

// ========================= DETECCIN DE RESPUESTAS CLAVE =========================
// ========================= DETECCIN DE RESPUESTAS CLAVE MEJORADA =========================
const analizarRespuestaCliente = (transcripcion) => {
  if (!transcripcion) return null;
  
  const texto = transcripcion.toLowerCase();
  
  // Deteccin de "ya pagu"
  if (texto.includes('ya pag') || texto.includes('realic el pago') || 
      texto.includes('pague') || texto.includes('cancel')) {
    return 'ya_pago';
  }
  
  // Deteccin de nmero equivocado
  if (texto.includes('nmero equivocado') || texto.includes('numero equivocado') ||
      texto.includes('no es') || texto.includes('quin es') || 
      texto.includes('quien es') || texto.includes('se equivoc')) {
    return 'numero_equivocado';
  }
  
  // Deteccin de encargado de pagos
  if (texto.includes('s') || texto.includes('si') || texto.includes('correcto') || 
      texto.includes('afirmativo') || texto.includes('yo soy')) {
    return 'confirmado';
  }
  
  if (texto.includes('no') || texto.includes('otra persona') || 
      texto.includes('mi espos') || texto.includes('esposa') || texto.includes('esposo')) {
    return 'no_encargado';
  }
  
  // DETECCIN DE SOLICITUDES DE TRANSFERENCIA - NUEVO
  if (texto.includes('transfier') || texto.includes('asesor') || 
      texto.includes('supervisor') || texto.includes('agente humano') ||
      texto.includes('agente real') || texto.includes('persona real') ||
      texto.includes('operador') || texto.includes('representante') ||
      texto.includes('quiero hablar con un') || texto.includes('necesito un humano') ||
      texto.includes('djame hablar con') || texto.includes('ponme con')) {
    return 'solicita_transferencia';
  }
  
  // Deteccin de motivos de no pago
  const motivos = [
    'incremento de factura', 'factura ms alta', 'aumento de la factura',
    'traslado no efectuado', 'mudanza', 'cambio de direccin',
    'posible fraude', 'fraude', 'cobro indebido',
    'cancelacin no atendida', 'quiero cancelar', 'no quiero el servicio',
    'sin recurso econmico', 'no tengo dinero', 'estoy sin trabajo',
    'inconsistencia con mtodo de pago', 'error en el pago', 'no pas el pago',
    'inconformidad con el servicio', 'mal servicio', 'no funciona'
  ];
  
  for (const motivo of motivos) {
    if (texto.includes(motivo)) return 'motivo_no_pago';
  }
  
  return null;
};

// ========================= NOTIFICACIN DE RESULTADOS =========================
const notificarResultadoLlamada = async (telefono, callSid, resultado, detalles = {}) => {
  try {
    await storage.set(`resultado:${callSid}`, {
      telefono,
      resultado,
      timestamp: new Date().toISOString(),
      detalles
    }, 3600);
    console.log(`?? Resultado notificado: ${telefono} - ${resultado} - ${convertirNumeroATexto(telefono)}`);
  } catch (error) {
    console.error('? Error notificando resultado:', error);
  }
};

const service = ({ logger, callService }) => {
  const svc = callService;

  svc.on('session:new', async (session) => {
    const { call_sid } = session;
    
    // Extraer el telfono de la URI SIP
    const v_to = session.to;
    const v_parts = v_to.split(':')[1].split('@')[0];
    const telefono = v_parts.startsWith('7757') ? v_parts.slice(4) : v_parts;
    
    session.locals = {
      logger: logger.child({ call_sid }),
      call_sid,
      telefono,
      startTime: new Date(),
      resultado: 'en_progreso',
      etapa: 'inicio',
      datosCliente: null
    };
    
    try {
      // Obtener datos del contacto
      const datos = await storage.get(`callinfo:${telefono}`);
      if (!datos) {
        session.locals.logger.error('No se encontraron datos del contacto en Redis');
        await notificarResultadoLlamada(telefono, call_sid, 'fallido', { error: 'datos_no_encontrados' });
        return session.hangup().send();
      }

      session.locals.identificacion = datos.identificacion;
      session.locals.datosContacto = datos;
      session.locals.datosCliente = datos;

      // Validar API Key
      const apiKey = process.env.ULTRAVOX_API_KEY;
      if (!apiKey) {
        session.locals.logger.error('ULTRAVOX_API_KEY missing, hanging up');
        await notificarResultadoLlamada(telefono, call_sid, 'fallido', { error: 'api_key_missing' });
        return session.hangup().send();
      }

      session.locals.logger.info({ session, datos }, `[CALL] Nueva llamada entrante: ${call_sid}`);
      yardMaster.addSession(call_sid);
      
      // Funciones auxiliares
      const parseDecimal = (str) => {
        if (typeof str !== 'string') return str;
        return parseFloat(str.replace(/,/g, '').replace(/\s+/g, '').trim());
      };

      const pronunciarDigitos = (numeroStr) => {
        return String(numeroStr).split('').join(', ');
      };

      // ========================= PROMPT MEJORADO CON FLUJO CLARO =========================
      const buildSystemPrompt = (datosContacto) => {
        const cliente = datosContacto.nombrecompleto || 'Estimado cliente';
        const monto = parseDecimal(datosContacto.monto_inicial) || 0;
        const montoEnLetras = convertirNumeroATexto(monto);
        
        // Mapeo de tipos de servicio
        const mapearTipoServicio = (origen) => {
          const mapeo = {
            'BSCS': 'Servicio Mvil',
            'bscs': 'Servicio Mvil',
            'ASCARD': 'Servicio Hogar',
            'ascard': 'Servicio Hogar', 
            'RR': 'Equipo Financiado',
            'rr': 'Equipo Financiado',
            'Servicio movil': 'Servicio Mvil',
            'servicio movil': 'Servicio Mvil'
          };
          return mapeo[origen] || origen || 'Servicio Claro';
        };

        const tipoServicio = mapearTipoServicio(datosContacto.origen);
        const plazoAcuerdo = datosContacto.plazo_acuerdo || 'prximos 5 das hbiles';
        const diasMora = datosContacto.dias_mora || '0';
        const min = datosContacto.min || '';
        const descuento = datosContacto.descuento || '';
        
        const calcularMontoConDescuento = () => {
          if (!descuento) return montoEnLetras;
          const descuentoNum = parseFloat(descuento) || 0;
          const montoConDescuento = monto * (1 - descuentoNum / 100);
          return convertirNumeroATexto(Math.round(montoConDescuento), 'dinero', 'Es-Co');
        };
        
      const construirMensajeDeuda = () => {
        const mensajeBase = `Le llamo acerca del pago pendiente de su ${tipoServicio.toLowerCase()}`;
          
        switch(tipoServicio) {
          case 'Servicio Hogar':
            return `${mensajeBase} con referencia ${convertirNumeroATexto(datosContacto.referencia)}, por un valor de $${montoEnLetras}, que se encuentra con ${diasMora} das de mora`;
          
          case 'Servicio Mvil':
            return `${mensajeBase} con nmero de Linea ${min}, por un valor de $${montoEnLetras}, que se encuentra con ${diasMora} das de mora`;
          
          case 'Equipo Financiado':
            return `${mensajeBase} con Plan ${datosContacto.plan || 'el plan contratado'}, por un valor de $${montoEnLetras}, que se encuentra con ${diasMora} das de mora`;
          
          default:
            return `${mensajeBase}, por un valor de $${montoEnLetras}, que se encuentra con ${diasMora} das de mora`;
        }
      };
      
      // Construir oferta con descuento si est disponible
      const construirOferta = () => {
        if (descuento) {
          return `Tenemos una oferta especial de descuento del ${descuento}% para que regularice su deuda de $${montoEnLetras}.`;
        } else {
          return `Para regularizar su situacin, necesita cancelar el valor total de $${montoEnLetras}.`;
        }
      };
      
       // OFERTA DE DESCUENTO MEJORADA
      const construirOfertaDescuento = () => {
          if (descuento) {
            const montoConDesc = calcularMontoConDescuento();
            return `Tenemos una oferta especial de descuento del ${descuento}% para que regularice su deuda. En lugar de $${montoEnLetras}, puede pagar solo $${montoConDesc} si realiza el pago dentro de ${plazoAcuerdo}.`;
          } else {
            return `Para regularizar su situacin, necesita cancelar el valor total de $${montoEnLetras} dentro de ${plazoAcuerdo}.`;
          }
        };
  
      const construirPreguntaEncargado = () => {
        switch(tipoServicio) {
          case 'Servicio Mvil':
             if (min) {
              return `Es usted el encargado de realizar los pagos de la lnea ${min}?`;
            } else {
              return `Es usted el encargado de realizar los pagos de la lnea ${telefono || 'su lnea mvil'}?`;
            }
          
          case 'Servicio Hogar':
            return `Es usted el encargado de los pagos del servicio Hogar con Linea ${datosContacto.referencia}?`;
          
          case 'Equipo Financiado':
            return `Es usted el responsable del pago del plan ${datosContacto.plan || 'el plan contratado'}?`;
          
          default:
            return `Es usted el encargado de realizar los pagos de este servicio?`;
        }
      };


        return `
ROLE: Eres Andrea, agente especializada en cobranza de Claro. Tu estilo es:
- Emptico pero profesional y directa
- Clara en la comunicacin
- Persuasiva pero no agresiva
- Orientada a soluciones
- Buena pronunciacin de nmeros

CONTEXTO:
Ests contactando a ${cliente} de parte de "ATM BPO Aliado estratgico para Claro". 
Le informa que esta llamada est siendo grabada y monitoreada.

DETALLE DE LA OBLIGACIN:
- Producto: ${tipoServicio}
- Nmero de cuenta: ${pronunciarDigitos(datosContacto.referencia)}
${min ? `- MIN/Nmero de lnea: ${min}` : ''}
- Valor adeudado: $${montoEnLetras}
${descuento ? `- Descuento disponible: ${descuento}% (Pagara solo $${calcularMontoConDescuento()})` : '- No hay descuentos disponibles en este momento'}
- Fecha de vencimiento: ${datosContacto.fecha_vencimiento || 'No especificada'}
- Das en mora: ${diasMora} das
- Fecha mxima acuerdo: ${plazoAcuerdo}
- Plan: ${datosContacto.plan || 'No especificado'}
- Referencia: ${datosContacto.referencia || 'No especificado'}

FLUJO DE COBRANZA CLARO - SIGUE ESTOS PASOS ESTRICTAMENTE:

ETAPA 1 - INICIO:
1. Saluda: "Buenos das/tardes, habla Andrea de ATM BPO, aliado estratgico para Claro"
2. Informa grabacin: "Le informo que esta llamada est siendo grabada y monitoreada"
3. Confirma identidad: "Me confirma si hablo con ${cliente.split(' ')[0] || 'el titular'}?"

ETAPA 2 - EXPOSICIN DE DEUDA:
4. Si NO es el titular: "Podra comunicarme con la persona encargada de los pagos del servicio?"
5. Si ES el titular: "${construirMensajeDeuda()}"

ETAPA 3 - CONFIRMACIN DE ENCARGADO:
6. Pregunta especfica: "${construirPreguntaEncargado()}"

ETAPA 4 - OFERTA DE ACUERDO CON DESCUENTO:
7. Si CONFIRMA ser encargado: "${construirOfertaDescuento()}"
8. Si hay descuento: ENFATIZAR el beneficio "Aproveche el ahorro del ${descuento}% pagando solo $${calcularMontoConDescuento()}"
9. Si NO hay descuento: "Necesita cancelar el valor total de $${montoEnLetras}"
10. Si acepta: "Perfecto, le enviaremos los detalles del acuerdo por SMS"
11. Si muestra inters pero duda: Reforzar beneficios del pago oportuno

ETAPA 5 - BENEFICIOS Y CONSECUENCIAS:
12. Segn el servicio, menciona:
    - SERVICIO MVIL: 
      * Beneficios: "Mantendr su servicio activo, buen historial de pago y score financiero"
      * Consecuencias: "Riesgo de suspensin del servicio y afectacin crediticia"
    
    - SERVICIO HOGAR:
      * Beneficios: "Conserva el servicio activo y evita cargos por reconexin"
      * Consecuencias: "Posible suspensin y cobro de reconexin de $45.339 con IVA"
    
    - EQUIPO FINANCIADO:
      * Beneficios: "Mantenga su equipo al da y mejore su historial crediticio"
      * Consecuencias: "Intereses moratorios y posibles reportes negativos"

MANEJO DE OBJECIONES ESPECFICAS:
- Si dice "YA PAG": Preguntar "Cundo realiz el pago?, a qu referencia?, por qu valor?, por qu medio?"
- Si dice "NMERO EQUIVOCADO": Confirmar y terminar cordialmente
- Si alega problemas: Derivar a especialista mencionando "Le transfiero con un especialista para revisar su caso"
- Si pregunta por DESCUENTOS: 
  ${descuento ? 
    `Informar: "S, tenemos un descuento del ${descuento}% disponible. En lugar de $${montoEnLetras}, pagara solo $${calcularMontoConDescuento()} si realiza el pago en ${plazoAcuerdo}"` : 
    'Informar: "En este momento no hay descuentos especiales disponibles para su caso, pero puede acceder a otros beneficios pagando oportunamente"'
  }
- Si pregunta "HAY PROMOCIONES?": Responder igual que para descuentos
- Si dice "NO PUEDO PAGAR TODO": 
  ${descuento ?
    `Ofrecer: "Con el descuento del ${descuento}% pagara solo $${calcularMontoConDescuento()}, le sera ms manageable?"` :
    'Ofrecer: "Podemos evaluar un plan de pagos parciales, le interesara esa opcin?"'
  }

TRANSFERENCIAS A AGENTE HUMANO - USAR HERRAMIENTA transferir_a_agente CUANDO:
- Si el cliente dice: "quiero hablar con un agente", "transfiereme a un asesor", "habla con tu supervisor", "quiero un agente humano", "necesito ayuda humana"
- Si el cliente insiste repetidamente en hablar con alguien ms
- Si el cliente expresa frustracin y pide supervisor
- Si hay problemas complejos que no puedes resolver

INSTRUCCIN PARA TRANSFERIR: Cuando detectes alguna de las situaciones anteriores, DEBES usar la herramienta transferir_a_agente inmediatamente.

CIERRE:
- Resumir acuerdo: "Le confirmo: ${descuento ? `acuerdo con ${descuento}% de descuento para el ${plazoAcuerdo} por $${calcularMontoConDescuento()}` : `acuerdo para el ${plazoAcuerdo} por $${montoEnLetras}`}, referencia ${datosContacto.referencia}"
- Despedida: "Quedamos atentos a su pago. Que tenga un buen da!"

INSTRUCCIONES TCNICAS:
- Los pagos se realizan SOLAMENTE por App Mi Claro o Portal de Pagos Claro
- Si detectas buzn de voz: terminar inmediatamente
- Mantn conversacin fluida y profesional
- Pronuncia nmeros con pausas naturales
- Si pide referencia: informar ${datosContacto.referencia}
- PARA TRANSFERIR: Usar la herramienta transferir_a_agente cuando el cliente lo solicite explcitamente
`.trim();
};
            
      const fullSystemPrompt = buildSystemPrompt(datos);

      // Configurar adaptador Ultravox
      const adaptor = new Adaptor(session.locals.logger, apiKey);
      session.locals.adaptor = adaptor;
     
      adaptor.on('ready', () => {
        session.locals.logger.info('? Adaptador Ultravox listo (ready)');
        adaptor.setOutgoingJambonzSocket(session.ws);
      });
      
      adaptor.setIncomingJambonzSocket(session.ws);
 
      // Configurar eventos
      session
        .on('/event', onEvent.bind(null, session))
        .on('/toolCall', onToolCall.bind(null, session))
        .on('/final', onFinal.bind(null, session))
        .on('/referAction', onReferAction.bind(null, session))      
        .on('/referEvent', onReferEvent.bind(null, session))        
        .on('/sip_referAction', sipReferAction.bind(null, session))
        .on('/sip_referEvent', sipReferEvent.bind(null, session))
        .on('call:status', onCallStatus.bind(null, session))
        .on('close', onClose.bind(null, session))
        .on('error', onError.bind(null, session));

      // Configurar sesin
      session.config({
        listen: {
          enable: true,
          url: `${process.env.WS_BASE_URL}/audio-stream?callSid=${call_sid}`,
          mixType: 'mono',
          sampleRate: 8000,
          bidirectionalAudio: {
            enabled: true,
            streaming: true,
            sampleRate: 8000
          }
        }
      });
      
      let toolsConfig = false;
      const callTransferMode = process.env.CALL_TRANSFER || 'None';
        if (callTransferMode !== 'None') {
          toolsConfig = {
            selectedTools: [
              {
                temporaryTool: {
                  modelToolName: 'call-transfer',
                  description: 'Transfiere la llamada a un especialista',
                  dynamicParameters: [
                    {
                      name: 'conversation_summary',
                      location: 'PARAMETER_LOCATION_BODY',
                      schema: {
                        type: 'string',
                        description: 'Un resumen de la conversacin hasta ahora'
                      },
                      required: true
                    }
                  ],
                  client: {}
                }
              }
            ]
          };
        }
          
      // Iniciar LLM con el flujo de cobranza
      session.llm({
        vendor: 'ultravox',
        model: 'fixie-ai/ultravox',
        auth: { apiKey },
        actionHook: '/final',
        eventHook: '/event',
        toolHook: '/toolCall',
        llmOptions: {
          systemPrompt: fullSystemPrompt,
          firstSpeaker: 'FIRST_SPEAKER_AGENT',
          initialMessages: [
            {
              medium: 'MESSAGE_MEDIUM_VOICE',
              role: 'MESSAGE_ROLE_AGENT',
              text: 'Hola, soy Andrea del rea de cobranza de Claro.',
            }
          ],
          model: 'fixie-ai/ultravox',
          voice: 'Andrea-Spanish',
          transcriptOptional: true,
          ...(toolsConfig ? toolsConfig : {})
        }
      }).hangup().send();
     
    } catch (err) {
      session.locals.logger.error({ err }, 'Error en session:new');
      await notificarResultadoLlamada(telefono, call_sid, 'fallido', { error: err.message });
      safeClose(session.locals.logger, session.call_sid);
      session.close();
    }
  });
};

// ========================= EVENTOS MEJORADOS =========================
const onEvent = async (session, evt) => {
  const { logger, call_sid, telefono, datosCliente } = session.locals;
  logger.info({ evt }, 'Evento recibido en /event');

  if (evt?.type === 'transcript' && evt?.final) {
    const turns = Array.isArray(evt.transcript)
      ? evt.transcript
      : [{ role: evt.role, text: evt.text ?? evt.message }];

    try {
      if (!session.locals.call_sid) {
        logger.warn({ evtKeys: Object.keys(evt || {}) }, 'transcript.final sin call_sid');
        return session.reply();
      }

      for (const t of turns) {
        const message = (t?.text ?? t?.message ?? '').trim();
        if (!message) continue;

        // Deteccin de buzn en tiempo real
        if (t.role === 'customer' && detectarBuzonVoz(message)) {
          logger.info(`?? Buzn detectado para ${telefono}, colgando llamada`);
          session.locals.resultado = 'buzon';
          
          await notificarResultadoLlamada(telefono, call_sid, 'buzon', {
            mensaje: message,
            accion: 'terminacion_automatica'
          });
          
          setTimeout(() => {
            session.hangup().send();
          }, 1000);
          break;
        }
        
        // Anlisis de respuestas del cliente
        if (t.role === 'customer') {
          const tipoRespuesta = analizarRespuestaCliente(message);
          if (tipoRespuesta) {
            logger.info(`?? Respuesta detectada: ${tipoRespuesta} - ${message.substring(0, 50)}...`);
            
            // Actualizar estado segn respuesta
            switch(tipoRespuesta) {
              case 'ya_pago':
                session.locals.etapa = 'ya_pago';
                break;
              case 'numero_equivocado':
                session.locals.etapa = 'equivocado';
                break;
              case 'confirmado':
                session.locals.etapa = 'encargado_confirmado';
                break;
              case 'motivo_no_pago':
                session.locals.etapa = 'derivacion_especialista';
                break;
                
               case 'solicita_transferencia': // NUEVO
                session.locals.etapa = 'solicita_transferencia';
                logger.info(`?? Cliente solicita transferencia: ${telefono}`);
                break;
            }
          }
        }
        
        // Guardar transcripcin
        await TranscriptManager.saveMessage(session.locals.call_sid, {
          speaker: (t?.role || 'USER').toUpperCase(),
          message,
          metadata: {
            etapa: session.locals.etapa,
            tipoRespuesta: t.role === 'customer' ? analizarRespuestaCliente(message) : null
          }
        });
      }
    } catch (err) {
      logger.error({ err, evtShape: Object.keys(evt || {}) }, 'Error guardando transcripcin completa');
    }
  }

  session.reply();
};


// ========================= TOOL CALL MEJORADO - SIN MLTIPLES INVITES =========================
/*const onToolCall = async (session, evt) => {
  const { logger, telefono, call_sid, datosCliente } = session.locals || {};
  // normalizar propiedades del evento (soportar varios SDKs/versions)
  const tool_call_id = evt.tool_call_id || evt.toolCallId || evt.invocation_id || null;
  const name = evt.name || evt.tool || null;
  // args o parameters (segn versin)
  const args = evt.args || evt.parameters || evt.arguments || {};
  const conversation_summary = args.conversation_summary || args.summary || '';

  logger.info({ name, args, tool_call_id }, 'Tool hook recibido');

  try {
    if (!name) {
      throw new Error('Tool call sin name');
    }

    // TRANSFERENCIA A AGENTE MEJORADA - SIN MLTIPLES INVITES
    if (['transferir_a_agente', 'call-transfer', 'call_transfer', 'callTransfer'].includes(name)) {
      logger.info(`? Solicitando transferencia a agente: ${telefono}`);

      const transferType = (process.env.TRANSFER_TYPE || 'Refer'); // 'Dial' o 'Refer'
      // Preferir TRANSFER_TO, si no usar HUMAN_AGENT_NUMBER
      const transferToRaw = process.env.TRANSFER_TO || process.env.HUMAN_AGENT_NUMBER;
      const transferFrom = process.env.TRANSFER_FROM || process.env.CALLER_ID_OVERRIDE || 'bot4tm';
      const transferCarrier = process.env.TRANSFER_CARRIER || null;
      const callTransferMode = (process.env.CALL_TRANSFER || 'None'); // 'None'|'Cold'|'Warm'

      if (!transferToRaw) {
        logger.error('TRANSFER_TO/HUMAN_AGENT_NUMBER no configurado');
        throw new Error('TRANSFER_TO no configurado');
      }

      const successOutput = {
        type: 'client_tool_result',
        invocation_id: tool_call_id,
        result: 'Transferencia a agente humano iniciada correctamente'
      };

      if (transferType === 'Dial') {
        // Para Dial: transferTo debe ser un nmero (preferible E.164)
        const dialNumber = transferToRaw;
        const confirmHook = (callTransferMode === 'Warm') ? { confirmHook: '/confirmAction' } : {};

        const dialObj = {
          verb: 'dial',
          actionHook: '/dialAction',
          callerId: transferFrom,
          dialMusic: 'https://jambonz.app/us_ringback.mp3',
          target: [
            {
              type: 'phone',
              number: dialNumber,
              ...(transferCarrier ? { trunk: transferCarrier } : {})
            }
          ],
          ...confirmHook
        };

        logger.info({ dialObj }, 'Enviando redirect -> dial');
        session.sendCommand('redirect', [dialObj]);

        if (tool_call_id) session.sendToolOutput(tool_call_id, successOutput);

      } else {
        // **CORRECCIN PRINCIPAL: Usar 'refer' en lugar de 'sip:refer' para evitar mltiples INVITES**
        const sipTrunk = process.env.SIP_TRUNK_DOMAIN || '';
        const referTo = transferToRaw.startsWith('sip:') ? transferToRaw : `sip:${transferToRaw}${sipTrunk ? `@${sipTrunk}` : ''}`;

        // **COMANDO REFER CORREGIDO - Evita crear nueva sesin**
        const referObj = {
          verb: 'refer',
          referTo: referTo,
          referReplaces: call_sid, // IMPORTANTE! Reemplaza la llamada actual
          headers: {
            'Referred-By': `<sip:${process.env.CALLER_ID_OVERRIDE}>`,
            'Replaces': `${call_sid};from-tag=${session.from_tag};to-tag=${session.to_tag}`,
            'Contact': `<sip:${transferFrom}${sipTrunk ? `@${sipTrunk}` : ''}>`
          },
          actionHook: '/referAction',
          eventHook: '/referEvent'
        };

        logger.info({ referObj }, 'Enviando refer -> transferencia SIP');
        session.sendCommand(referObj);

        if (tool_call_id) session.sendToolOutput(tool_call_id, successOutput);
      }

      // Guardar resumen si viene (para confirmAction en Warm)
      if (conversation_summary) session.locals.conversation_summary = conversation_summary;

      // Intentar notificar resultado (no bloquear flujo)
      try {
        session.locals.resultado = 'transferido';
        await notificarResultadoLlamada(telefono, call_sid, 'transferido', {
          agente: transferToRaw,
          motivo: args?.motivo || 'solicitud_cliente',
          metodo: transferType === 'Dial' ? 'dial' : 'sip_refer'
        });
      } catch (errNotify) {
        logger.warn({ errNotify }, 'No crtico: error notificando transferencia');
      }

      return;
    }

    // **NUEVA HERRAMIENTA: Transferencia directa sin confirmacin**
    if (name === 'transferencia_directa' || name === 'transfer_immediate') {
      logger.info(`? Transferencia directa solicitada: ${telefono}`);

      const transferTo = process.env.TRANSFER_TO || process.env.HUMAN_AGENT_NUMBER;
      if (!transferTo) {
        throw new Error('TRANSFER_TO no configurado');
      }

      const sipTrunk = process.env.SIP_TRUNK_DOMAIN || '';
      const referTo = transferTo.startsWith('sip:') ? transferTo : `sip:${transferTo}${sipTrunk ? `@${sipTrunk}` : ''}`;

      // Transferencia inmediata sin interaccin adicional
      const referObj = {
        verb: 'refer',
        referTo: referTo,
        referReplaces: call_sid,
        headers: {
          'Referred-By': `<sip:${process.env.CALLER_ID_OVERRIDE}>`,
          'Replaces': `${call_sid};from-tag=${session.from_tag};to-tag=${session.to_tag}`
        }
      };

      logger.info({ referObj }, 'Enviando transferencia directa');
      session.sendCommand(referObj);

      // Notificacin inmediata
      session.locals.resultado = 'transferido_directo';
      await notificarResultadoLlamada(telefono, call_sid, 'transferido', {
        agente: transferTo,
        motivo: 'transferencia_automatica',
        metodo: 'sip_refer_directo'
      });

      if (tool_call_id) {
        session.sendToolOutput(tool_call_id, {
          type: 'client_tool_result',
          invocation_id: tool_call_id,
          result: 'Transferencia directa completada'
        });
      }

      return;
    }

    // Otros tool names (ejemplos que ya manejas)
    if (name === 'terminar_llamada_equivocada') {
      logger.info(`? Terminando por nmero equivocado: ${telefono}`);
      session.locals.resultado = 'equivocado';
      try { 
        await notificarResultadoLlamada(telefono, call_sid, 'equivocado'); 
      } catch (e) { 
        logger.error({ e }, 'Error notificando numero equivocado'); 
      }
      setTimeout(() => session.hangup().send(), 2000);
      return;
    }

    if (name === 'derivar_a_especialista') {
      logger.info(`?? Derivando a especialista: ${telefono}`);
      session.locals.resultado = 'derivado';
      try { 
        await notificarResultadoLlamada(telefono, call_sid, 'derivado', { 
          motivo: args?.motivo || 'motivo_no_pago' 
        }); 
      } catch (e) { 
        logger.error({ e }, 'Error notificando derivacion'); 
      }
      
      // Opcional: Implementar transferencia a especialista especfico
      const especialistaNumber = process.env.ESPECIALISTA_NUMBER || process.env.HUMAN_AGENT_NUMBER;
      if (especialistaNumber && tool_call_id) {
        const sipTrunk = process.env.SIP_TRUNK_DOMAIN || '';
        const referTo = especialistaNumber.startsWith('sip:') ? especialistaNumber : `sip:${especialistaNumber}${sipTrunk ? `@${sipTrunk}` : ''}`;
        
        const referObj = {
          verb: 'refer',
          referTo: referTo,
          referReplaces: call_sid,
          headers: {
            'Referred-By': `<sip:${process.env.CALLER_ID_OVERRIDE}>`,
            'Replaces': `${call_sid};from-tag=${session.from_tag};to-tag=${session.to_tag}`,
            'X-Reason': 'derivacion_especialista'
          }
        };
        
        session.sendCommand(referObj);
        session.sendToolOutput(tool_call_id, { 
          type: 'client_tool_result', 
          invocation_id: tool_call_id, 
          result: 'Derivado a especialista' 
        });
      } else if (tool_call_id) {
        session.sendToolOutput(tool_call_id, { 
          type: 'client_tool_result', 
          invocation_id: tool_call_id, 
          result: 'Derivado a especialista' 
        });
      }
      return;
    }

    if (name === 'registrar_acuerdo_pago') {
      logger.info(`? Acuerdo registrado: ${telefono}`);
      session.locals.resultado = 'acuerdo_registrado';
      try { 
        await notificarResultadoLlamada(telefono, call_sid, 'acuerdo_registrado', { 
          monto: args?.monto, 
          fecha: args?.fecha 
        }); 
      } catch (e) { 
        logger.error({ e }, 'Error notificando acuerdo pago'); 
      }
      if (tool_call_id) {
        session.sendToolOutput(tool_call_id, { 
          type: 'client_tool_result', 
          invocation_id: tool_call_id, 
          result: 'Acuerdo registrado' 
        });
      }
      return;
    }

    // **NUEVO: Manejo de solicitudes de supervisor**
    if (name === 'transferir_a_supervisor' || name === 'escalar_a_supervisor') {
      logger.info(`?? Escalando a supervisor: ${telefono}`);
      
      const supervisorNumber = process.env.SUPERVISOR_NUMBER || process.env.HUMAN_AGENT_NUMBER;
      if (!supervisorNumber) {
        throw new Error('SUPERVISOR_NUMBER no configurado');
      }

      const sipTrunk = process.env.SIP_TRUNK_DOMAIN || '';
      const referTo = supervisorNumber.startsWith('sip:') ? supervisorNumber : `sip:${supervisorNumber}${sipTrunk ? `@${sipTrunk}` : ''}`;

      const referObj = {
        verb: 'refer',
        referTo: referTo,
        referReplaces: call_sid,
        headers: {
          'Referred-By': `<sip:${process.env.CALLER_ID_OVERRIDE}>`,
          'Replaces': `${call_sid};from-tag=${session.from_tag};to-tag=${session.to_tag}`,
          'X-Escalation-Reason': 'solicitud_supervisor',
          'X-Conversation-Summary': conversation_summary.substring(0, 200) // Limitar longitud
        }
      };

      logger.info({ referObj }, 'Enviando escalacin a supervisor');
      session.sendCommand(referObj);

      session.locals.resultado = 'transferido_supervisor';
      await notificarResultadoLlamada(telefono, call_sid, 'transferido', {
        agente: supervisorNumber,
        motivo: 'solicitud_supervisor',
        metodo: 'sip_refer'
      });

      if (tool_call_id) {
        session.sendToolOutput(tool_call_id, {
          type: 'client_tool_result',
          invocation_id: tool_call_id,
          result: 'Transferido a supervisor correctamente'
        });
      }
      return;
    }

    // Default response for other tools
    if (tool_call_id) {
      session.sendToolOutput(tool_call_id, { 
        type: 'client_tool_result', 
        invocation_id: tool_call_id, 
        result: 'Operacin realizada correctamente' 
      });
    } else {
      logger.warn('tool_call_id ausente en toolCall por defecto');
    }

  } catch (err) {
    logger.error({ err }, 'Error en tool call');
    if (evt && (evt.tool_call_id || evt.toolCallId || tool_call_id)) {
      const id = tool_call_id || evt.tool_call_id || evt.toolCallId;
      session.sendToolOutput(id, { 
        type: 'client_tool_result', 
        invocation_id: id, 
        error_message: `Error procesando la herramienta: ${err.message}` 
      });
    }
  }
};*/

// ========================= HANDLERS ADICIONALES PARA TRANSFERENCIAS =========================
const onReferAction = async (session, evt) => {
  const { logger, telefono } = session.locals;
  logger.info({ evt }, '[REFER] Action recibido');
  
  // Manejar respuestas del comando refer
  if (evt.status === 'completed') {
    logger.info(`? Transferencia por REFER completada para ${telefono}`);
  } else if (evt.status === 'failed') {
    logger.error(`? Transferencia por REFER fallida para ${telefono}: ${evt.reason}`);
    
    // Opcional: Reintentar con mtodo alternativo
    if (evt.reason?.includes('no answer') || evt.reason?.includes('busy')) {
      logger.info(`? Reintentando transferencia para ${telefono} con mtodo alternativo`);
      // Aqu podras implementar lgica de reintento
    }
  }
};

const onReferEvent = async (session, evt) => {
  const { logger, telefono } = session.locals;
  logger.info({ evt }, '[REFER] Event recibido');
  
  // Manejar eventos NOTIFY de transferencias REFER
  if (evt.sip_status) {
    logger.info(`? Estado SIP en transferencia: ${evt.sip_status} para ${telefono}`);
  }
};




const onToolCall = async (session, evt) => {
  const { logger, telefono, call_sid, datosCliente } = session.locals || {};
  // normalizar propiedades del evento (soportar varios SDKs/versions)
  const tool_call_id = evt.tool_call_id || evt.toolCallId || evt.invocation_id || null;
  const name = evt.name || evt.tool || null;
  // args o parameters (segn versin)
  const args = evt.args || evt.parameters || evt.arguments || {};
  const conversation_summary = args.conversation_summary || args.summary || '';

  logger.info({ name, args, tool_call_id }, 'Tool hook recibido');

  try {
    if (!name) {
      throw new Error('Tool call sin name');
    }

    // TRANSFERENCIA A AGENTE (nombres esperados)
    if (['transferir_a_agente', 'call-transfer', 'call_transfer', 'callTransfer'].includes(name)) {
      logger.info(`? Solicitando transferencia a agente: ${telefono}`);

      const transferType = (process.env.TRANSFER_TYPE || 'Refer'); // 'Dial' o 'Refer'
      // Preferir TRANSFER_TO, si no usar HUMAN_AGENT_NUMBER
      const transferToRaw = process.env.TRANSFER_TO || process.env.HUMAN_AGENT_NUMBER;
      const transferFrom = process.env.TRANSFER_FROM || process.env.CALLER_ID_OVERRIDE || 'bot4tm';
      const transferCarrier = process.env.TRANSFER_CARRIER || null;
      const callTransferMode = (process.env.CALL_TRANSFER || 'None'); // 'None'|'Cold'|'Warm'

      if (!transferToRaw) {
        logger.error('TRANSFER_TO/HUMAN_AGENT_NUMBER no configurado');
        throw new Error('TRANSFER_TO no configurado');
      }

      const successOutput = {
        type: 'client_tool_result',
        invocation_id: tool_call_id,
        result: 'Transferencia a agente humano iniciada correctamente'
      };

      if (transferType === 'Dial') {
        // Para Dial: transferTo debe ser un nmero (preferible E.164)
        const dialNumber = transferToRaw;
        const confirmHook = (callTransferMode === 'Warm') ? { confirmHook: '/confirmAction' } : {};

        const dialObj = {
          verb: 'dial',
          actionHook: '/dialAction',
          callerId: transferFrom,
          dialMusic: 'https://jambonz.app/us_ringback.mp3',
          target: [
            {
              type: 'phone',
              number: dialNumber,
              ...(transferCarrier ? { trunk: transferCarrier } : {})
            }
          ],
          ...confirmHook
        };

        logger.info({ dialObj }, 'Enviando redirect -> dial');
        session.sendCommand('redirect', [dialObj]);

        if (tool_call_id) session.sendToolOutput(tool_call_id, successOutput);

      } else {
        // REFER path: construir SIP URI si transferToRaw no viene en formato sip:
        const sipTrunk = process.env.SIP_TRUNK_DOMAIN || '';
        const referTo = transferToRaw.startsWith('sip:') ? transferToRaw : `sip:${transferToRaw}${sipTrunk ? `@${sipTrunk}` : ''}`;

        logger.info({ referTo }, 'Enviando redirect -> sip:refer');
        session.sendCommand('redirect', [{ verb: 'sip:refer', referTo }]);

        if (tool_call_id) session.sendToolOutput(tool_call_id, successOutput);
      }

      // Guardar resumen si viene (para confirmAction en Warm)
      if (conversation_summary) session.locals.conversation_summary = conversation_summary;

      // Intentar notificar resultado (no bloquear flujo)
      try {
        session.locals.resultado = 'transferido';
        await notificarResultadoLlamada(telefono, call_sid, 'transferido', {
          agente: transferToRaw,
          motivo: args?.motivo || 'solicitud_cliente'
        });
      } catch (errNotify) {
        logger.warn({ errNotify }, 'No crtico: error notificando transferencia');
      }

      return;
    }

    // Otros tool names (ejemplos que ya manejas)
    if (name === 'terminar_llamada_equivocada') {
      logger.info(`? Terminando por nmero equivocado: ${telefono}`);
      session.locals.resultado = 'equivocado';
      try { await notificarResultadoLlamada(telefono, call_sid, 'equivocado'); } catch (e) { logger.error({ e }, 'Error notificando numero equivocado'); }
      setTimeout(() => session.hangup().send(), 2000);
      return;
    }

    if (name === 'derivar_a_especialista') {
      logger.info(`?? Derivando a especialista: ${telefono}`);
      session.locals.resultado = 'derivado';
      try { await notificarResultadoLlamada(telefono, call_sid, 'derivado', { motivo: args?.motivo || 'motivo_no_pago' }); } catch (e) { logger.error({ e }, 'Error notificando derivacion'); }
      if (tool_call_id) session.sendToolOutput(tool_call_id, { type: 'client_tool_result', invocation_id: tool_call_id, result: 'Derivado a especialista' });
      return;
    }

    if (name === 'registrar_acuerdo_pago') {
      logger.info(`? Acuerdo registrado: ${telefono}`);
      session.locals.resultado = 'acuerdo_registrado';
      try { await notificarResultadoLlamada(telefono, call_sid, 'acuerdo_registrado', { monto: args?.monto, fecha: args?.fecha }); } catch (e) { logger.error({ e }, 'Error notificando acuerdo pago'); }
      if (tool_call_id) session.sendToolOutput(tool_call_id, { type: 'client_tool_result', invocation_id: tool_call_id, result: 'Acuerdo registrado' });
      return;
    }

    // Default response for other tools
    if (tool_call_id) {
      session.sendToolOutput(tool_call_id, { type: 'client_tool_result', invocation_id: tool_call_id, result: 'Operacin realizada correctamente' });
    } else {
      logger.warn('tool_call_id ausente en toolCall por defecto');
    }

  } catch (err) {
    logger.error({ err }, 'Error en tool call');
    if (evt && (evt.tool_call_id || evt.toolCallId || tool_call_id)) {
      const id = tool_call_id || evt.tool_call_id || evt.toolCallId;
      session.sendToolOutput(id, { type: 'client_tool_result', invocation_id: id, error_message: 'Error procesando la herramienta' });
    }
  }
}; 


/*const onToolCall = async (session, evt) => {
  const { logger, telefono, call_sid, datosCliente } = session.locals || {};
  const { tool_call_id, name, parameters } = evt || {};
  logger.info({ name, parameters }, 'Tool hook recibido');

  try {
    // TRANSFERENCIA A AGENTE
    if (name === 'transferir_a_agente' || name === 'call-transfer') {
      logger.info(`? Solicitando transferencia a agente: ${telefono}`);

      const referTo = process.env.HUMAN_AGENT_NUMBER;
      if (!referTo) {
        logger.error('HUMAN_AGENT_NUMBER no configurado');
        throw new Error('HUMAN_AGENT_NUMBER no configurado en variables de entorno');
      }

      const sipTrunk = process.env.SIP_TRUNK_DOMAIN;
      if (!sipTrunk) {
        logger.error('SIP_TRUNK_DOMAIN no configurado');
        throw new Error('SIP_TRUNK_DOMAIN no configurado en variables de entorno');
      }

      const sipUri = `sip:${referTo}@${sipTrunk}`;
      logger.info(`? Transferiendo a: ${sipUri}`);

      try {
        session.sendCommand('redirect', [
          {
            verb: 'sip:refer',
            referTo: sipUri
          }
        ]);

        if (tool_call_id) {
          session.sendToolOutput(tool_call_id, {
            type: 'client_tool_result',
            invocation_id: tool_call_id,
            result: 'Transferencia a agente humano iniciada correctamente'
          });
        } else {
          logger.warn('No tool_call_id presente; no se enviar toolOutput');
        }

        session.locals.resultado = 'transferido';
        try {
          await notificarResultadoLlamada(telefono, call_sid, 'transferido', {
            agente: referTo,
            motivo: parameters?.motivo || 'solicitud_cliente'
          });
        } catch (errNotify) {
          logger.error({ errNotify }, 'Error notificando resultado de transferencia (no crtico)');
        }

        return;
      } catch (err) {
        logger.error({ err }, 'Error ejecutando redirect');
        if (tool_call_id) {
          session.sendToolOutput(tool_call_id, {
            type: 'client_tool_result',
            invocation_id: tool_call_id,
            error_message: 'No se pudo iniciar la transferencia'
          });
        }
        return;
      }
    }

    // TERMINAR POR NMERO EQUIVOCADO
    if (name === 'terminar_llamada_equivocada') {
      logger.info(`? Terminando por nmero equivocado: ${telefono}`);
      session.locals.resultado = 'equivocado';
      try {
        await notificarResultadoLlamada(telefono, call_sid, 'equivocado');
      } catch (e) {
        logger.error({ e }, 'Error notificando numero equivocado');
      }

      setTimeout(() => {
        session.hangup().send();
      }, 2000);
      return;
    }

    // DERIVAR A ESPECIALISTA
    if (name === 'derivar_a_especialista') {
      logger.info(`?? Derivando a especialista: ${telefono}`);
      session.locals.resultado = 'derivado';
      try {
        await notificarResultadoLlamada(telefono, call_sid, 'derivado', {
          motivo: parameters?.motivo || 'motivo_no_pago'
        });
      } catch (e) {
        logger.error({ e }, 'Error notificando derivacion');
      }
      return;
    }

    // REGISTRAR ACUERDO DE PAGO
    if (name === 'registrar_acuerdo_pago') {
      logger.info(`? Acuerdo registrado: ${telefono}`);
      session.locals.resultado = 'acuerdo_registrado';
      try {
        await notificarResultadoLlamada(telefono, call_sid, 'acuerdo_registrado', {
          monto: parameters?.monto,
          fecha: parameters?.fecha
        });
      } catch (e) {
        logger.error({ e }, 'Error notificando acuerdo pago');
      }

      if (tool_call_id) {
        session.sendToolOutput(tool_call_id, {
          type: 'client_tool_result',
          invocation_id: tool_call_id,
          result: 'Acuerdo registrado'
        });
      }
      return;
    }

    // Respuesta por defecto para otras herramientas
    if (tool_call_id) {
      session.sendToolOutput(tool_call_id, {
        type: 'client_tool_result',
        invocation_id: tool_call_id,
        result: 'Operacin realizada correctamente'
      });
    } else {
      logger.warn('tool_call_id ausente en toolCall por defecto');
    }

  } catch (err) {
    logger.error({ err }, 'Error en tool call');
    if (tool_call_id) {
      session.sendToolOutput(tool_call_id, {
        type: 'client_tool_result',
        invocation_id: tool_call_id,
        error_message: 'Error procesando la herramienta'
      });
    }
  }
};

*/

// ========================= onFinal MEJORADO =========================
const onFinal = async (session, evt) => {
  const { logger, telefono, call_sid, resultado, etapa } = session.locals;
  logger.info({ evt }, 'Evento final recibido');
  
  // Guardar transcripcin final
  if (evt.transcript && Array.isArray(evt.transcript)) {
    try {
      for (const turn of evt.transcript) {
        await TranscriptManager.saveMessage(session.call_sid, {
          speaker: turn.role === 'agent' ? 'AGENT' : 
                  turn.role === 'customer' ? 'CUSTOMER' : 'BOT',
          message: turn.text || turn.message || '',
          timestamp: new Date(turn.timestamp || Date.now()),
          metadata: { etapa_final: etapa }
        });
      }
      logger.info(`?? Guardada transcripcin final con ${evt.transcript.length} mensajes`);
    } catch (error) {
      logger.error({ error }, 'Error guardando transcripcin final');
    }
  }

  // Notificar resultado final
  const resultadoFinal = resultado === 'en_progreso' ? 'completado' : resultado;
  await notificarResultadoLlamada(telefono, call_sid, resultadoFinal, {
    duracion: Date.now() - session.locals.startTime,
    etapa_final: etapa,
    transcripcion_completa: evt.transcript ? 'si' : 'no'
  });

  session.reply();
};

// ========================= FUNCIONES RESTANTES =========================
const onCallStatus = (session, evt) => {
  const { logger, telefono, call_sid } = session.locals;
  logger.info({ evt }, 'Estado de llamada');
  
  if (evt.call_status === 'failed' || evt.call_status === 'no-answer') {
    notificarResultadoLlamada(telefono, call_sid, 'fallido', {
      estado: evt.call_status,
      motivo: evt.reason || 'desconocido'
    });
  }
  
  if (!session.locals.call_sid_b && evt.direction === 'outbound') {
    session.locals.call_sid_b = evt.call_sid;
  }
};

const sipReferAction = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, '[SIP] REFER Action');
};

const sipReferEvent = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, '[SIP] REFER Event');
};

const onClose = (session, code, reason) => {
  const { logger, telefono, call_sid, resultado } = session.locals;
  logger.info({ code, reason }, 'Sesin cerrada');
  
  if (resultado === 'en_progreso') {
    const resultadoCierre = code === 1000 ? 'completado' : 'fallido';
    notificarResultadoLlamada(telefono, call_sid, resultadoCierre, {
      codigo_cierre: code,
      razon: reason
    });
  }
  
  detenerEjecucionLlamadas();
  safeClose(logger, session.call_sid);
};

const onError = (session, err) => {
  const { logger, telefono, call_sid } = session.locals;
  logger.error({ err }, 'Error en sesin');
  
  notificarResultadoLlamada(telefono, call_sid, 'fallido', {
    error: err.message,
    tipo: 'error_sesion'
  });
  
  safeClose(logger, session.call_sid);
};

module.exports = service;