// call-handler.js
const fs = require('fs');
const path = require('path');
const csv = require('csv-parser');

// ====== CARGAR CONTACTOS AL INICIAR ======
const CONTACTS_FILE = path.join(__dirname, '../../data/contactos.csv');
let contactosMap = new Map();

(function cargarContactos() {
  fs.createReadStream(CONTACTS_FILE)
    .pipe(csv())
    .on('data', (row) => {
      const tel = row.telefono.replace(/\D/g, '').replace(/^\+/, '');
      contactosMap.set(tel, row);
    })
    .on('end', () => console.log('✅ Contactos cargados:', contactosMap.size))
    .on('error', (err) => console.error('❌ Error cargando contactos:', err));
})();

module.exports = ({ logger, makeService }) => {
  const svc = makeService({ path: '/call' });

  svc.on('session:new', (session) => {
    session.locals = { logger: logger.child({ call_sid: session.call_sid }) };
    logger.info({ session }, `Nueva llamada entrante: ${session.call_sid}`);

    const from = session.from.replace(/\D/g, '').replace(/^\+/, '');
    const contacto = contactosMap.get(from);

    if (!contacto) {
      session.say({ text: 'No se encontró el contacto en la base de datos.' }).hangup().send();
      return;
    }

    session.locals.contacto = contacto;
    const apiKey = process.env.ULTRAVOX_API_KEY;

    try {
      session
        .on('/event', onEvent.bind(null, session))
        .on('/final', onFinal.bind(null, session))
        .on('close', onClose.bind(null, session))
        .on('error', onError.bind(null, session))
        .on('/toolCall', onToolCall.bind(null, session))
        .on('/dialAction', dialAction.bind(null, session));

      session
        .answer()
        .pause({ length: 1.5 })
        .llm({
          vendor: 'ultravox',
          model: 'fixie-ai/ultravox',
          auth: { apiKey },
          actionHook: '/final',
          eventHook: '/event',
          toolHook: '/toolCall',
          llmOptions: {
            systemPrompt: 'Eres una agente de cobranza llamada Andrea. Hablas en español, con tono directo pero cordial.',
            firstSpeaker: 'FIRST_SPEAKER_AGENT',
            initialMessages: [{
              medium: 'MESSAGE_MEDIUM_VOICE',
              role: 'MESSAGE_ROLE_USER'
            }],
            model: 'fixie-ai/ultravox',
            voice: 'Andrea-Spanish',
            transcriptOptional: true,
            selectedTools: [
              {
                temporaryTool: {
                  modelToolName: 'call-transfer',
                  description: 'Transfiere la llamada a un agente humano',
                  client: {}
                }
              }
            ]
          }
        })
        .send();
    } catch (err) {
      session.locals.logger.info({ err }, `Error respondiendo llamada: ${session.call_sid}`);
      session.close();
    }
  });
};

const onEvent = async (session, evt) => {
  session.locals.logger.info(`Evento recibido: ${JSON.stringify(evt)}`);
};

const onFinal = async (session, evt) => {
  const logger = session.locals.logger;
  logger.info(`Final de interacción: ${JSON.stringify(evt)}`);
  if (["server failure", "server error"].includes(evt.completion_reason)) {
    const msg = evt.error?.code === 'rate_limit_exceeded'
      ? 'Lo sentimos, has superado el límite de uso. Intenta en unos minutos.'
      : 'Hubo un error procesando tu solicitud.';
    session.say({ text: msg }).hangup();
  }
  session.reply();
};

const onClose = (session, code, reason) => {
  session.locals.logger.info({ code, reason }, `Sesion cerrada: ${session.call_sid}`);
};

const onError = (session, err) => {
  session.locals.logger.info({ err }, `Error en sesion: ${session.call_sid}`);
};

const onToolCall = async (session, evt) => {
  const logger = session.locals.logger;
  const { tool_call_id } = evt;
  const contacto = session.locals.contacto;
  const numero = contacto.telefono.replace(/\D/g, '').replace(/^\+/, '');

  try {
    session.sendToolOutput(tool_call_id, {
      type: 'client_tool_result',
      invocation_id: tool_call_id,
      result: 'Transfiriendo a un agente humano. Por favor espera...'
    });

    setTimeout(() => {
      session.sendCommand('redirect', [
        { 
          verb: 'say', 
          text: 'Conectando con un agente humano. Por favor espere.' },
        {
          verb: 'dial',
          actionHook: '/dialAction',
          callerId: 'bot4tm',
          target: [
            {
              type: 'sip',
              sipUri: `sip:7757${numero}@atmbponextcall.controlnextapp.com`,
              trunk: process.env.HUMAN_AGENT_TRUNK
            }
          ]
        }
      ]);
    }, 3000);

  } catch (err) {
    logger.info({ err }, 'Error al transferir la llamada');
    session.sendToolOutput(tool_call_id, {
      type: 'client_tool_result',
      invocation_id: tool_call_id,
      error_message: 'Error al transferir llamada'
    });
  }
};

const dialAction = async (session, evt) => {
  session.locals.logger.info('Llamada con agente humano finalizada.');
  session.say({ text: 'La llamada con el agente humano ha finalizado. Gracias.' }).hangup().reply();
};
