// file: routes/ws/call-handler.js
const { detenerEjecucionLlamadas } = require('../../utils/call-launcher');
const safeClose = require('../../utils/safe-close');
const yardMaster = require('../../utils/yardmaster');
const { WebhookResponse } = require('@jambonz/node-client');
const Adaptor = require('../../utils/ultravox_s2s');
const TranscriptManager = require('../../utils/transcript-manager');
const storage = require('../../utils/storage');
const convertirNumeroATexto = require('../../utils/numero-a-texto');


const service = ({ logger, callService }) => {
  const svc = callService;

  svc.on('session:new', async (session) => {
    const { call_sid } = session;
    
    // Extraer el teléfono de la URI SIP
    const v_to = session.to;
    const v_parts = v_to.split(':')[1].split('@')[0];
    const telefono = v_parts.startsWith('7757') ? v_parts.slice(4) : v_parts;
    
    session.locals = {
      logger: logger.child({ call_sid }),
      call_sid,
      telefono
    };
    
   
    try {
    
      const datos = await storage.get(`callinfo:${telefono}`);
      if (!datos) {
        session.locals.logger.error('No se encontraron datos del contacto en Redis');
        return session.hangup().send();
      }

      session.locals.identificacion = datos.identificacion;
      session.locals.datosContacto = datos;

      //const adaptor = new Adaptor(session.locals.logger, process.env.ULTRAVOX_API_KEY);
      //session.locals.adaptor = adaptor;
      
      const apiKey = process.env.ULTRAVOX_API_KEY;
      if (!apiKey) {
        session.locals.logger.error('ULTRAVOX_API_KEY missing, hanging up');
        return session.hangup().send();
      }

      session.locals.logger.info({ session, datos }, `[CALL] Nueva llamada entrante: ${call_sid}`);
      yardMaster.addSession(call_sid);
      
      // Construir el prompt con los datos del CSV (similar a call-handler31082025.js)
      const parseDecimal = (str) => {
        if (typeof str !== 'string') return str;
        return parseFloat(str.replace(/,/g, '').replace(/\s+/g, '').trim());
      };

      const pronunciarDigitos = (numeroStr) => {
        return String(numeroStr).split('').join(', ');
      };

      const buildSystemPrompt = (datosContacto) => {
        const cliente = datosContacto.nombrecompleto || 'Estimado cliente';
        const cuenta = datosContacto.cuenta || 'XXXXXXXX';
        const cuentaEnDigitos = cuenta.split('').join(', ');
        const monto = parseDecimal(datosContacto.monto_inicial) || 0;
        const montoEnLetras = convertirNumeroATexto(monto);

        return `
      ROLE: Eres Andrea, agente especializada en cobranza de Claro. Tu estilo es:
      - Empático pero profesional y directa
      - Clara en la comunicación
      - Persuasiva pero no agresiva
      - Orientada a soluciones
      - Buena pronunciación de números
      
      CONTEXTO:
      Estás contactando a ${cliente} cuyo número de cuenta es: ${cuentaEnDigitos}, quien tiene una obligación vencida por un total de $${montoEnLetras}.
      
      DETALLE DE LA OBLIGACIÓN:
      - Producto: ${datosContacto.origen || 'No especificado'} (${datosContacto.tipo_obligacion || 'No especificado'})
      - Número de cuenta: ${pronunciarDigitos(cuenta)}
      - Valor adeudado: $${montoEnLetras}
      - Fecha de vencimiento: ${datosContacto.fecha_vencimiento || 'No especificada'}
      - Días en mora: ${datosContacto.dias_mora || '0'} días
      - Plan: ${datosContacto.plan || 'No especificado'}
      - Referencia: ${datosContacto.referencia || 'No especificado'}
      
      INSTRUCCIONES:
      1. Saluda cordialmente y confirma que hablas con ${cliente.split(' ')[0] || 'el cliente'}
      2. Menciona el motivo de tu llamada: "Le llamo acerca de su obligación vencida con Claro"
      3. Informa el monto adeudado: $${montoEnLetras}
      4. Menciona los días de mora: ${datosContacto.dias_mora} días
      5. Ofrece alternativas de pago (pago total)
      6. Usa lenguaje sencillo y evita términos técnicos
      7. Los pagos se realizan solamente por el app o la web de Claro
      8. Mantén la conversación fluida y profesional
      9. Cuando leas números, usa pausas naturales
      10. Si el cliente te pide la referencia o te aprueba el pago reconociendo los valores, informa el numero de referencia ${datosContacto.referencia}
      `.trim();
            };
            
      const fullSystemPrompt = buildSystemPrompt(datos);

      const adaptor = new Adaptor(session.locals.logger, apiKey);
      session.locals.adaptor = adaptor;
     
     adaptor.on('ready', () => {
        session.locals.logger.info(' Adaptador Ultravox listo (ready)');
       adaptor.setOutgoingJambonzSocket(session.ws); // si ya tienes este ws
      });
      
      // Cuando recibas audio desde Jambonz
     adaptor.setIncomingJambonzSocket(session.ws); // o desde tu WebSocket handler
 
      session
        .on('/event', onEvent.bind(null, session))
        .on('/toolCall', onToolCall.bind(null, session))
        .on('/final', onFinal.bind(null, session))
        .on('/sip_referAction', sipReferAction.bind(null, session))
        .on('/sip_referEvent', sipReferEvent.bind(null, session))
        .on('call:status', onCallStatus.bind(null, session))
        .on('close', onClose.bind(null, session))
        .on('error', onError.bind(null, session));

      session.config({
        listen: {
          enable: true,
          url: `${process.env.WS_BASE_URL}/audio-stream?callSid=${call_sid}`,
          mixType: 'mono',
          sampleRate: 8000,
          bidirectionalAudio: {
            enabled: true,
            streaming: true,
            sampleRate: 8000
          }
        }
      });

      session.llm({
        vendor: 'ultravox',
        model: 'fixie-ai/ultravox',
        auth: { apiKey },
        actionHook: '/final',
        eventHook: '/event',
        toolHook: '/toolCall',
        llmOptions: {
          systemPrompt: fullSystemPrompt,
          firstSpeaker: 'FIRST_SPEAKER_AGENT',
          initialMessages: [
            {
              medium: 'MESSAGE_MEDIUM_VOICE',
              role: 'MESSAGE_ROLE_AGENT',
              text: 'Hola, soy del área de cobranza.',
            }
          ],
          model: 'fixie-ai/ultravox',
          voice: 'Andrea-Spanish',
          transcriptOptional: true,
        }
      }).send();
     
    } catch (err) {
      session.locals.logger.error({ err }, 'Error en session:new');
      safeClose(session.locals.logger, session.call_sid);
      session.close();
    }
  });
};

// === EVENTOS ===

const onEvent = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, 'Evento recibido en /event');

  if (evt?.type === 'transcript' && evt?.final) {
    // 1) Normaliza: puede venir como array o como un solo turno
    const turns = Array.isArray(evt.transcript)
      ? evt.transcript
      : [{ role: evt.role, text: evt.text ?? evt.message }];

    try {
      if (!session.locals.call_sid) {
        logger.warn({ evtKeys: Object.keys(evt || {}) }, 'transcript.final sin call_sid');
        return session.reply();
      }

      for (const t of turns) {
        const message = (t?.text ?? t?.message ?? '').trim();
        if (!message) continue; // evita inserts vacíos o null
        await TranscriptManager.saveMessage(session.locals.call_sid, {
          speaker: (t?.role || 'USER').toUpperCase(),
          message
        });
      }
    } catch (err) {
      // Con pino, usa la key "err" para que imprima stack y message
      logger.error({ err, evtShape: Object.keys(evt || {}) }, 'Error guardando transcripción completa');
    }
  }

  session.reply();
};


const onToolCall = async (session, evt) => {
  const { logger } = session.locals;
  const { tool_call_id } = evt;
  logger.info({ evt }, ' Tool hook recibido');

  try {
    const data = {
      type: 'client_tool_result',
      invocation_id: tool_call_id,
      result: 'Operación realizada correctamente',
    };
    session.sendToolOutput(tool_call_id, data);
  } catch (err) {
    logger.error({ err }, 'Error enviando resultado de herramienta');
  }
};

const onFinal = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, 'Evento final recibido');
  
  // Guardar transcripción final si existe
  if (evt.transcript && Array.isArray(evt.transcript)) {
    try {
      for (const turn of evt.transcript) {
        await TranscriptManager.saveMessage(session.call_sid, {
          speaker: turn.role === 'agent' ? 'AGENT' : 
                  turn.role === 'customer' ? 'CUSTOMER' : 'BOT',
          message: turn.text || turn.message || '',
          timestamp: new Date(turn.timestamp || Date.now())
        });
      }
      logger.info(`Guardada transcripción final con ${evt.transcript.length} mensajes`);
    } catch (error) {
      logger.error({ error }, 'Error guardando transcripción final');
    }
  }
  session.reply();
};

const onCallStatus = (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, 'Estado de llamada');
  if (!session.locals.call_sid_b && evt.direction === 'outbound') {
    session.locals.call_sid_b = evt.call_sid;
  }
};

const sipReferAction = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, '[SIP] REFER Action');
};

const sipReferEvent = async (session, evt) => {
  const { logger } = session.locals;
  logger.info({ evt }, '[SIP] REFER Event');
};

const onClose = (session, code, reason) => {
  const { logger } = session.locals;
  logger.info({ code, reason }, ' Sesión cerrada');
 // session.locals.adaptor?.close?.();
  detenerEjecucionLlamadas();
  safeClose(logger, session.call_sid);
};

const onError = (session, err) => {
  const { logger } = session.locals;
  logger.error({ err }, 'Error en sesión');
 // session.locals.adaptor?.close?.();
  safeClose(logger, session.call_sid);
};

module.exports = service;
