require('dotenv').config();
const fs = require('fs');
const path = require('path');
const jambonz = require('@jambonz/node-client');
const logger = require('./logger')('call-launcher');
const storage = require('./storage');

const atmService = require('./../services/atm-service');
const { consultarObligaciones, obtenerTokenATM } = atmService;

const client = jambonz(process.env.JAMBONZ_ACCOUNT_SID, process.env.JAMBONZ_API_KEY, {
  baseUrl: process.env.JAMBONZ_BASE_URL
});

const CONTACTS_FILE = path.join(__dirname, '../config/contactos_temp.json');
let detenerLlamadas = false;

  const yaFueLlamadoExitosamente = async (numero) => {
    return await storage.get(`call_success:${numero}`);
  };

  const marcarComoLlamadoExitosamente = async (numero) => {
    await storage.set(`call_success:${numero}`, true, { ttl: 60 * 60 * 24 }); // 24h opcional
  };

const normalizePhone = (raw) => {
  
  let phone = String(raw).replace(/[^\d]/g, '');
  if (phone.startsWith('57') && phone.length === 12) phone = phone.substring(2);
  return phone;
};

const launchCall = async (telefono, identificacion) => {
  const numero = normalizePhone(telefono);

  // Guarda información básica de la llamada
  await storage.set(`callinfo:${numero}`, {
    numero,
    identificacion,
    timestamp: Date.now(),
    status: 'pending'
  });

  // Marca como pendiente (esperando saber si fue contestada realmente)
  await storage.set(`call_pending:${numero}`, true, { ttl: 60 * 10 }); // 10 minutos

  const payload = {
    application_sid: process.env.APPLICATION_SID,
    from: 'bot4tm',
    to: {
      type: 'sip',
      sipUri: `sip:7757${numero}@atmbponextcall.controlnextapp.com`
    },
    call_hook: {
      url: `${process.env.WS_BASE_URL}/call`,
      method: 'GET'
    },
    call_status_hook: {
      url: `${process.env.HTTP_BASE_URL}/status`,
      method: 'POST'
    },
    speech_synthesis_vendor: 'google',
    speech_synthesis_language: 'es-US',
    speech_synthesis_voice: 'es-US-Standard-A',
    speech_recognizer_vendor: 'google',
    speech_recognizer_language: 'es-CO'
  };

  try {
    const response = await client.calls.create(payload);
    logger.info(` Llamando a ${numero} - call_sid: ${response.sid}`);
    return response.sid;
  } catch (err) {
    // Si falla la llamada, borra callinfo y pendiente
    await storage.del(`callinfo:${numero}`);
    await storage.del(`call_pending:${numero}`);
    
    logger.error(` Error al llamar a ${numero}`, {
      errorMessage: err.message,
      statusCode: err.status,
      details: err.response?.data || err.response || err
    });
    throw err;
  }
};

/*const launchCall = async (telefono, identificacion) => {
  const numero = normalizePhone(telefono);

   await storage.set(`callinfo:${numero}`, {
    numero,
    identificacion,
    timestamp: Date.now(),
    status: 'pending'
  });

  const payload = {
    application_sid: process.env.APPLICATION_SID,
    from: 'bot4tm',
    to: {
      type: 'sip',
      sipUri: `sip:7757${numero}@atmbponextcall.controlnextapp.com`
    },
    call_hook: {
      url: `${process.env.WS_BASE_URL}/call`,
      method: 'GET'
    },
    call_status_hook: {
      url: `${process.env.HTTP_BASE_URL}/status`,
      method: 'POST'
    },
    speech_synthesis_vendor: 'google',
    speech_synthesis_language: 'es-US',
    speech_synthesis_voice: 'es-US-Standard-A',
    speech_recognizer_vendor: 'google',
    speech_recognizer_language: 'es-CO'
  };

  try {
     //logger.info('Payload a enviar:', payload);
    const response = await client.calls.create(payload);
    logger.info(`Llamando a ${numero}`);
    console.log("call_sid cllalanunchge:", response.sid);
    return response.sid;
  } catch (err) {
    await storage.del(`callinfo:${numero}`);
    logger.error(`Error al llamar a ${numero}`, {
      errorMessage: err.message,
      statusCode: err.status,
      details: err.response?.data || err.response || err
    });
    throw err;
  }
}; */

const ejecutarLlamadas = async () => {
  if (!fs.existsSync(CONTACTS_FILE)) {
    logger.warn('No se encontró el archivo JSON de contactos.');
    return;
  }

  const contactos = JSON.parse(fs.readFileSync(CONTACTS_FILE, 'utf8'));

  for (const contacto of contactos) {
    if (detenerLlamadas) break;

    let { telefono, identificacion } = contacto;
    if (!telefono || !identificacion) {
      logger.warn(`Contacto inválido: ${telefono} | ${identificacion}`);
      continue;
    }
    
    const dTel = String(telefono).replace(/[^\d]/g, '');
    const dId  = String(identificacion).replace(/[^\d]/g, '');

    const numero = normalizePhone(telefono);
    


    if (await yaFueLlamadoExitosamente(numero)) {
      logger.info(`${numero} ya fue llamado exitosamente, se omite.`);
      continue;
    }

    try {
      const token = await obtenerTokenATM();
      const obligaciones = await consultarObligaciones(String(identificacion).replace(/[^\d]/g, ''));

      if (!Array.isArray(obligaciones) || obligaciones.length === 0) {
        logger.warn(`No se encontraron obligaciones para ${identificacion} (${numero}), se omite la llamada.`);
        continue; // pasa al siguiente contacto
      }

      await storage.set(`obligaciones:precheck:${numero}`, obligaciones);
      const callSid = await launchCall(telefono, identificacion);

      logger.info(`Esperando confirmación de conversación con ${numero}...`);

      // Esperar hasta 2 minutos a que desaparezca call_pending
      const MAX_ESPERA_MS = 2 * 60 * 1000;
      const INTERVALO_MS = 5000;
      let intentos = Math.ceil(MAX_ESPERA_MS / INTERVALO_MS);
      let confirmacion = false;

      while (intentos-- > 0) {
        const pendiente = await storage.get(`call_pending:${numero}`);
        const fueExitosa = await yaFueLlamadoExitosamente(numero);

        if (!pendiente || fueExitosa) {
          confirmacion = fueExitosa;
          break;
        }

        await new Promise(res => setTimeout(res, INTERVALO_MS));
      }

      if (confirmacion) {
        logger.info(`Confirmada conversación con ${numero}`);
      } else {
        logger.warn(`No se logró confirmar conversación real con ${numero} en el tiempo límite`);
      }

      await new Promise(res => setTimeout(res, 10000)); // Pausa antes del siguiente intento
    } catch (err) {
      logger.error({ err, telefono }, 'Error al ejecutar llamada individual');
    }
  }
};


/*const ejecutarLlamadas = async () => {
  if (!fs.existsSync(CONTACTS_FILE)) {
    logger.warn('No se encontró el archivo JSON de contactos.');
    return;
  }

  const contactos = JSON.parse(fs.readFileSync(CONTACTS_FILE, 'utf8'));
  
  for (const contacto of contactos) {
    if (detenerLlamadas) break;

    const { telefono, identificacion } = contacto;
    if (!telefono || !identificacion) {
      logger.warn(`Contacto inválido: ${telefono} | ${identificacion}`);
      continue;
    }

    const numero = normalizePhone(telefono);

    if (await yaFueLlamadoExitosamente(numero)) {
      logger.info(`Número ${numero} ya fue llamado exitosamente, se omite.`);
      continue;
    }

    try {
      await launchCall(telefono, identificacion);
      await new Promise(res => setTimeout(res, 30000));
    } catch (err) {
      logger.error({ err, telefono }, 'Error en llamada individual');
    }
  }
};*/

const detenerEjecucionLlamadas = () => {
  detenerLlamadas = true;
  logger.warn('Se detuvo la ejecución de llamadas.');
};

module.exports = {
  ejecutarLlamadas,
  detenerEjecucionLlamadas
};